import java.util.*;
import java.io.*;

/** A class to store a word and its frequency in a language.
*
* @author Scott MacKenzie, 2001-2005
*/
public class Word implements Comparable

{
   protected String word;
   protected int freq;

   /** Construct a Word object.
   * @param w a String representing a word
   * @param f an int representing the frequency of the word in a language
   */
   public Word(String w, int f)
   {
      word = w;
      freq = f;
   }

   public String getWord() { return word; }
   public int getFreq() { return freq; }
	public void incFreq() { freq += 1; }
   public String toString() { return word + " " + freq; }

   /** Load a dictionary file into a <code>Word</code> array.
   *
   * A dictionary file contains a series of lines, each containing
   * two white-space delimited entries: a word and the frequency
   * of the word in the target language.<p>
   *
   * As an example of a dictionary file, here are a few lines from
   * the file <code>d1-wordfreq.txt</code>:<p>
   *
   * <pre>
   *     any      108043
   *     only     105411
   *     people   102516
   *     than     101495
   *     should   99069
   *     like     87862
   *     between  87034
   *     very     86823
   * </pre>
   *
   * @param fileName the name of the dictionary file
   * @return a <code>Word</code> array.
   *
   */
   public static Word[] loadDictionary(String fileName) throws IOException 
   {
      // ensure dictionary file exists
      File f = new File(fileName);
      if (!f.exists())
      {
         System.out.println("File not found - " + fileName);
         System.exit(1);
      }

      // open dictionary file for input
      BufferedReader inputFile =
         new BufferedReader(new FileReader(fileName));

      // read lines from dictionary ('word' and 'frequency' on each)
      String line;
      Vector<Word> v = new Vector<Word>();
      while ((line = inputFile.readLine()) != null)
      {
         StringTokenizer st = new StringTokenizer(line);

         // exactly two entries per line required
         if (st.countTokens() != 2)
         {
            System.out.println("Dictionary format error");
            System.exit(1);
         }

         // get the word
         String newWord = st.nextToken();

         // get the frequency
         int newFreq = Integer.parseInt(st.nextToken());

			// brief modification to read French word-freq list which contains
			// reals, instead of integers, for the word frequencies
         //int newFreq = (int)((Double.parseDouble(st.nextToken())) * 100.0);

         // add to vector as a Word object
         v.addElement(new Word(newWord, newFreq));
      }

      // close disk file
      inputFile.close(); 

      // declare an EncodedWord array of just the size needed
      Word[] dict = new Word[v.size()];

      // copy elements from vector into array
      v.copyInto(dict);

      // sort the dictionary by code
      //Arrays.sort(dict, new ByCode());
      return dict;
   }

	public boolean equals(Word w)
	{
		return this.getWord().equals(w.getWord());
	}

	public int compareTo(Object o)
	{
		return this.getWord().compareTo(((Word)o).getWord());
	}

}
/*
class ByWord implements Comparator
{
   public int compare(Object o1, Object o2)
   {
      String s1 = ((Word)o1).getWord();
      String s2 = ((Word)o2).getWord();
      return s1.compareTo(s2);
   }
}

class ByFreq implements Comparator
{
   public int compare(Object o1, Object o2)
   {
      double d1 = ((Word)o1).getFreq();
      double d2 = ((Word)o2).getFreq();
      return (int)(d2 - d1);
   }
}

*/
