import java.io.*;
import java.util.*;

/** A program to build a keystrokes-appended version of a
 * word-frequency, digram-frequency, or letter-frequency file.<p>
 *
 * Output is written to stdout.  Use this program to create the various
 * keystroke versions of the word-, digram-, and letter-frequency files
 * for subsequent processing by <code>KSPCWords</code>, <code>KSPCDigrams</code>,
 * or <code>KSPCLetters</code>.  For example, to generate the multitap
 * KSPC statistics for RIM's <i>SureType</i> keyboard, the following commands
 * may be used:
 * <p>
 *
 * <pre>
 *     PROMPT>java BuildKeystrokes d1-wordfreq.txt -wf -mtst > temp.txt
 *     PROMPT>java KSPCWords temp.txt
 *     KSPC26 = 1.4952118809351231 KSPC27 = 1.4039626335925184
 * </pre>
 *
 * For the "basic encoding" modes, additional processing is needed if T9-like entry
 * is of interest.  For example, to generate the KSPC statistics for 
 * RIM's <i>SureType</i> keyboard for T9-like entry, 
 * the following commands could be used used:<p>
 *
 * <pre>
 *     PROMPT>java BuildKeystrokes d1-wordfreq.txt -wf -st > temp.txt
 *     PROMPT>java T9 temp.txt -k > temp1.txt
 *     PROMPT>java KSPCWords temp1.txt
 *     KSPC26 = 1.0023919872082925 KSPC27 = 1.0019512323701862
 * </pre>
 *
 * Usage:<p>
 *
 * <pre>
 *     PROMPT>java BuildKeystrokes file -ss -kk
 *
 *     where file = a word-freq, digram-freq, or letter-freq file
 *           -ss = source file format (see below)
 *           -kk = keystroke format (see below)
 * </pre>
 *
 * Source file formats:
 *
 * <pre>
 *     -wf = word frequency
 *     -df = digram frequency
 *     -lf = letter frequency
 * </pre>
 *
 * Keystroke formats:
 *
 * <pre>
 *     -mt6 = multitap (<a href=LettersOnSixKeys.gif>6 keys</a>)
 *     -mt4 = multitap (<a href=LettersOnFourKeys.gif>4 keys</a>)
 *     -mt2 = multitap (<a href=LettersOnTwoKeys.gif>2 keys</a>)
 *     -mt1 = multitap (1 key!)
 *     -mttmk2 = multitap <a href=TouchMeKey4.jpg>TouchMeKey4</a> (Tanaka-Ishii et al., 2002)
 *     -mtst = multiap on RIM <a href=SureType.jpg>SureType</a>
 *     -mtstm = multiap on RIM <a href=SureType.jpg>SureType</a> (modified 2nd row: A SD FG HJ KL)
 *     -mteq3 = multitap on Eatoni <a href=eatoni-eq3.jpg>EQ3</a>
 *     -mteq6 = multitap on Eatoni <a href=eatoni-eq6.jpg>EQ6</a>
 *     -mthl = multitap on Hwang and Lee (2005) <a href=hwang-chi2005.jpg>Qwerty-like layout</a>
 *     -mtle = multitap on <a href=LetterEase.jpg>LetterEase</a> (Ryu & Cruz, 2005)
 *     -mtflpk = multitap on Fewer Letters per Key (<a href=flpk.jpg>FLpK</a>) (Ryu & Cruz, 2005)
 *     -mtlt = multitap on <a href=LessTap.jpg>LessTap</a> (Pavlovych & Stuerzlinger, 2003)
 *     -mtstick = multitap on the <a href=stick-CHI2004.jpg>Stick</a> (Green et al., 2004)
 *     -mtacd = multitap on Alphabetically-constrained design (Gong & Tarasewich, 2005)
 *     -k6 = letters on 6 keys (basic encoding)
 *     -k4 = letters on 4 keys (basic encoding)
 *     -k2 = letters on 2 keys (basic encoding)
 *     -k1 = letters on 1 key! (basic encoding)
 *     -tmk4 = TouchMeKey4 (Tanaka-Ishii et al., 2002) (basic encoding)
 *     -st = SureType (basic encoding)
 *     -stM = SureType modified (basic encoding with modified 2nd row: A SD FG HJ KL)
 *     -eq3 = Eatoni EQ3 (basic encoding)
 *     -eq6 = Eatoni EQ6 (basic encoding)
 *     -hl = Hwang and Lee (2005) Qwerty-like (basic encoding)
 *     -le = LetterEase (Ryu and Cruz, 2005) (basic encoding)
 *     -flpk = Fewer Letters per Key (FLpK) (Ryu & Cruz, 2005) (basic encoding)
 *     -lt = LessTap (Pavlovych & Stuerzlinger, 2003) (basic encoding)
 *     -stick = Stick (Green et al., 2004) (basic encoding)
 *     -acd = Alphabetically-Constrainded Design (Gong & Tarasewich, 2005) (basic encoding)
 *     -akko = AKKO (Harbusch & Kuhn, 2003)
 *     -pk = phone keypad (basic encoding)
 *     -mt = multitap on phone keypad
 *     -t9 = T9 (not supported, use T9.java)
 *     -lw = LetterWise (not supported, use Eatoni exe's)
 *     -me = MessageEase
 *     -qw = Qwerty
 *     -pg = pager
 *     -d1 = date stamp method #1
 *     -d2 = date stamp method #2
 *     -d3 = date stamp method #3
 *     -d4 = date stamp method #4
 *     -d7 = date stamp method #7
 *     -tt = two thumb method
 *     -twc = TiltWriter custom (Android)
 * </pre>
 *
 * Here is an example invocation:
 *
 * <pre>
 *
 *     PROMPT>java BuildKeystrokes d1-wordfreq.txt -wf -mt
 *     the 5776384 84433S
 *     of 2789403 666333S
 *     and 2421302 2663S
 *     .
 *     . (etc.)
 *     .
 *     on 590305 666N66S
 *     .
 *     . (etc.)
 *     .
 * </pre>
 *
 * @see <a href="BuildKeystrokes.java">source code</a>
 * @author Scott MacKenzie, 2002-2011
 */
public class BuildKeystrokes
{
	public static void main(String[] args) throws IOException
	{
		if (args.length != 3)
		{
			System.out.println(
					"usage: java BuildKeystrokes file -ss -kk\n" +
							"\n" +
							"where file = a word-freq, digram-freq, or letter-freq file\n" +
							"-ss = source file format (see below)\n" +
							"-kk = keystroke format (see below)\n" +
							"\n" +
							"Source file formats:\n" +
							"    -wf = word frequency\n" +
							"    -df = digram frequency\n" +
							"    -lf = letter frequency\n" +
							"\n" +
							"Keystroke formats:\n" +
							"    -mt6 = multitap with 6 keys + space\n" +
							"    -mt4 = multitap with 4 keys + space\n" +
							"    -mt2 = multitap with 2 keys + space\n" +
							"    -mt1 = multitap with 1 key + space\n" +
							"    -mttmk4 = multitap on TouchMeKey4 (Tanaka-Ishii et al., 2002)\n" +
							"    -mtst = multitap on RIM SureType (2nd row: AS DF GH JK L)\n" +
							"    -mtstm = multitap on RIM SureType modified (2nd row: A SD FG HJ KL)\n" +
							"    -mteq3 = multitap on Eatoni EQ3\n" +
							"    -mteq6 = multitap on Eatoni EQ6\n" +
							"    -mthl = multitap on Hwang and Lee (2005) Qwerty-like layout\n" +
							"    -mtle = multitap on LetterEase (Ryu & Cruz, 2005)\n" +
							"    -mtflpk = multitap on Fewer Letters per Key (FLpK) (Ryu & Cruz, 2005)\n" +
							"    -mtlt = multitap on LessTap (Pavlovych & Stuerzlinger, 2003)\n" +
							"    -mtstick = multitap on the Stick (Green et al., 2004)\n" +
							"    -mtacd = multitap on the Alphabetically-Constrained Design (Gong & Tarasewich, 2005)\n" +
							"    -mtakko = AKKO (Harbusch & Kuhn, 2003)\n" +
							"    -k6 = letters on 6 keys (basic encoding)\n" +
							"    -k4 = letters on 4 keys (basic encoding)\n" +
							"    -k2 = letters on 2 keys (basic encoding)\n" +
							"    -k1 = letters on 1 key! (basic encoding)\n" +
							"    -tmk4 = TouchMeKey4 (Tanaka-Ishii et al., 2002)\n" +
							"    -st = SureType (basic encoding; 2nd row: AS DF GH JK L)\n" +
							"    -stm = SureType modified (basic encoding; 2nd row: A SD FG HJ KL)\n" +
							"    -eq3 = Eatoni EQ3 (basic encoding)\n" +
							"    -eq6 = Eatoni EQ6 (basic encoding)\n" +
							"    -hl = Hwang and Lee (2005) Qwerty-like (basic encoding)\n" +
							"    -le = LetterEase (Ryu & Cruz, 2005)\n" +
							"    -flpk = Fewer Letter per Key (FLpK) (Ryu & Cruz, 2005)\n" +
							"    -lt = LessTap (Pavlovych & Stuerzlinger, 2003)\n" +
							"    -stick = Stick (Green et al., 2004)\n" +
							"    -acd = Alphabetically-Constrained Desing (Gong & Tarasewich, 2005)\n" +
							"    -pk = phone keypad (basic encoding)\n" +
							"    -mt = multitap\n" +
							"    -t9 = T9 (not supported, use T9.java)\n" +
							"    -lw = LetterWise (not supported, use Eatoni exe's)\n" +
							"    -me = MessageEase\n" +
							"    -qw = Qwerty\n" +
							"    -pg = pager\n" +
							"    -dn = date stamp method #n (n = 1, 2, 3, 4, or 7)\n" +
							"    -tt = two thumb method\n" +
							"    -twc = for TiltWriter custom (Android)"
					);
			return;
		}

		// convert command-line arguments to variables
		String inFile = args[0];
		//String charFile = args[1];

		// make sure file1 exists
		File f = new File(inFile);
		if (!f.exists())
		{
			System.out.println("File not found: " + inFile);
			System.exit(1);
		}

		// open dictionary file for input
		BufferedReader inputFile = new BufferedReader(new FileReader(inFile));

		// get character definition string
		//String characters = (new BufferedReader(new FileReader(charFile))).readLine();
		String characters = "_abcdefghijklmnopqrstuvwxyz";

		// Note: Last two characters above are Finnish (a with two dots,
		// followed by o with two dots

		// input file formats 
		boolean wordFreq = false;
		boolean digramFreq = false;
		boolean letterFreq = false;

		// keystroke format
		boolean mt6 = false;  // 6-key input (1=abcde 2=fghij 3=klmno 4=pqrst 5=uvwxy 6=z)
		boolean mt4 = false;  // 4-key input (1=abcdefg 2=hijklmn 3=opqrstu 4=vwxyz)
		boolean mt2 = false;  // 2-key input (1=abcdefghijklm 2=nopqrstuvwxyz)
		boolean mt1 = false;  // 1-key input (1 = abcdefghijklmnopqrstuvwxyz)
		boolean mttmk4 = false; // TouchMeKey4 (Tanaka-Ishii et al., 2002)
		boolean mtst = false; // multitap on RIM SureType
		//boolean mtstm = false; // multitap on RIM SureType modified
		boolean mteq3 = false; // multitap on Eatoni EQ3
		boolean mteq6 = false; // multitap on Eatoni EQ6
		boolean mthl = false; // multitap on Hwang and Lee (2005) Qwerty-like layout
		boolean mtle = false; // multitap on LetterEase (Ryu & Cruz, 2005)
		boolean mtflpk = false; // multitap on Fewer Letters per Key (FLpK) (Ryu & Cruz, 2005)
		boolean mtlt = false; // multitap on LessTap (Pavlovych & Stuerzlinder, 2003)
		boolean mtstick = false; // multitap on the Stick (Green et al., 2004)
		boolean mtacd = false; // multitap on the Alphabetically-Constrained Design (Gong & Tarasewich, 2005)
		boolean mtakko = false; // multitap on the AkKo design
		boolean k6 = false; // letters on 6 keys (basic encoding)
		boolean k4 = false; // letters on 4 keys (basic encoding)
		boolean k2 = false; // letters on 2 keys (basic encoding)
		boolean k1 = false; // letters on 1 key! (basic encoding)
		boolean tmk4 = false; // TouchMeKey4 (Tanaka-Ishii et al., 2002) (basic encoding)
		boolean sureType = false; // RIM SureType (basic encoding)
		boolean sureTypeModified = false; // RIM SureType modified (basic encoding)
		boolean eq3 = false; // Eatoni EQ3 (basic encoding)
		boolean eq6 = false; // Eatoni EQ6 (basic encoding)
		boolean hl = false; // Hwang and Lee (2005) (basic encoding)
		boolean le = false; // LetterEase (Ryu & Cruz, 2005) (basid encoding)
		boolean flpk = false; // Fewer Letters per Key (FLpK) (Ryu & Cruz, 2005) (basic encoding)
		boolean lt = false; // LessTap (Pavlovych & Stuerzlinger, 2003) (basic encoding)
		boolean stick = false; // Stick (Green et al., 2004) (basic encoding)
		boolean acd = false; // Alphabetically-Constrained Design (Gong & Tarasewich, 2005)
		boolean akko = false;
		boolean phoneKeypad = false;
		boolean multitap = false;
		boolean multitapOptimized = false;
		boolean t9 = false;
		boolean letterWise = false;
		boolean messagEase = false;
		boolean qwerty = false;
		boolean pager = false;
		boolean dateStamp1 = false;
		boolean dateStamp2 = false;
		boolean dateStamp3 = false;
		boolean dateStamp4 = false;
		boolean dateStamp7 = false;
		boolean twoThumb = false;
		boolean twc = false; // TiltWriter custom 

		if (args[1].equals("-wf")) wordFreq = true;
		if (args[1].equals("-df")) digramFreq = true;
		if (args[1].equals("-lf")) letterFreq = true;

		if (args[2].equals("-twc")) twc = true;
		if (args[2].equals("-mt6")) mt6 = true;
		if (args[2].equals("-mt4")) mt4 = true;
		if (args[2].equals("-mt2")) mt2 = true;
		if (args[2].equals("-mt1")) mt1 = true;
		if (args[2].equals("-mttmk4")) mttmk4 = true;
		if (args[2].equals("-mtst")) mtst = true;
		//if (args[2].equals("-mtstm")) mtstm = true;
		if (args[2].equals("-mteq3")) mteq3 = true;
		if (args[2].equals("-mteq6")) mteq6 = true;
		if (args[2].equals("-mthl")) mthl = true;
		if (args[2].equals("-mtle")) mtle = true;
		if (args[2].equals("-mtflpk")) mtflpk = true;
		if (args[2].equals("-mtlt")) mtlt = true;
		if (args[2].equals("-mtstick")) mtstick = true;
		if (args[2].equals("-mtacd")) mtacd = true;
		if (args[2].equals("-mtakko")) mtakko = true;
		if (args[2].equals("-k6")) k6 = true;
		if (args[2].equals("-k4")) k4 = true;
		if (args[2].equals("-k2")) k2 = true;
		if (args[2].equals("-k1")) k1 = true;
		if (args[2].equals("-tmk4")) tmk4 = true;
		if (args[2].equals("-st")) sureType = true;
		if (args[2].equals("-stm")) sureTypeModified = true;
		if (args[2].equals("-eq3")) eq3 = true;
		if (args[2].equals("-eq6")) eq6 = true;
		if (args[2].equals("-hl")) hl = true;
		if (args[2].equals("-le")) le = true;
		if (args[2].equals("-flpk")) flpk = true;
		if (args[2].equals("-lt")) lt = true;
		if (args[2].equals("-stick")) stick = true;
		if (args[2].equals("-acd")) acd = true;
		if (args[2].equals("-akko")) akko = true;
		if (args[2].equals("-pk")) phoneKeypad = true;
		if (args[2].equals("-mt")) multitap = true;
		if (args[2].equals("-t9")) t9 = true;
		if (args[2].equals("-lw")) letterWise = true;
		if (args[2].equals("-me")) messagEase = true;
		if (args[2].equals("-qw")) qwerty = true;
		if (args[2].equals("-pg")) pager = true;
		if (args[2].equals("-d1")) dateStamp1 = true;
		if (args[2].equals("-d2")) dateStamp2 = true;
		if (args[2].equals("-d3")) dateStamp3 = true;
		if (args[2].equals("-d4")) dateStamp4 = true;
		if (args[2].equals("-d7")) dateStamp7 = true;
		if (args[2].equals("-tt")) twoThumb = true;

		// Multitap SureType codes
		String[] mtstcode = {
				"?",   "8",  "7",  "4", "1", "44", "5", "55", "33", "6",
				"66",  "'",  "9", "88", "*", "**", "!", "11", "??",  "2",
				"3",  "77", "!!", "~~", "22", "~"
		};

		// Multitap SureType Modified codes
		// "modified" means the 2nd row is adjusted as follows: A SD FG HJ KL
		String[] mtstmcode = {
				"?",   "8",  "7",  "44", "1", "5", "55", "6", "33", "66",
				"'",  "''",  "9", "88", "*", "**", "!", "11", "4",  "2",
				"3",  "77", "!!", "~~", "22", "~"
		};


		// Multitap EQ3 codes
		String[] mteq3code = {
				"4", "8", "777", "5", "111", "55", "99", "6", "222",
				"66", "666", "*", "9", "88", "3", "33", "1", "1111",
				"44", "2", "**", "7777", "11", "77", "22", "7"
		};

		// Multitap HL codes
		String[] mthlcode = {
				"4", "888", "77", "44", "1", "22", "55", "8", "6", "666", "999", "66", "99",
				"9", "3", "33", "111", "5", "7", "2", "88", "555", "11", "777", "222", "444"
		};

		// Multitap LE codes
		String[] mtlecode = {
				"5", "77", "88", "#", "1", "66", "55", "0", "8", "##", "33", "11", "44",
				"7", "2", "**", "444", "6", "3", "4", "*", "00", "22", "99", "9", "111"
		};

		// Multitap FLPK codes
		String[] mtflpkcode = {
				"1", "11", "2", "22", "3", "33", "4", "44", "5", "55", "6", "66", "7",
				"77", "8", "88", "9", "99", "*", "**", "0", "00", "000", "#", "##", "###"	
		};

		// Multitap LessTap codes
		String[] mtltcode = {
				"2", "222", "2", "33", "3", "333", "444", "44", "4", "555", "55", "5", "666",
				"66", "6", "777", "7777", "77", "7", "8", "88", "888", "9", "999", "99", "9"
		};

		// Multitap Stick codes
		String[] mtstickcode = {
				"11", "555", "333", "33", "3", "44", "55", "66", "8",
				"77", "88", "99", "777", "666", "9", "999", "1", "4",
				"22", "5", "7", "444", "2", "222", "6", "111"
		};

		// Multitap Alphabetically-Constrained Design
		String[] mtacdcode = {
				"2", "22", "222", "2222", "3", "33", "333",
				"4", "44", "444", "4444", "44444", "5", "55",
				"6", "7", "77", "777", "7777", "8", "88", "888",
				"9", "99", "999", "9999"
		};

		// Multitap AKKO design
		String[] mtakkocode = {
				"2", "1", "3", "22", "33", "222", "333", "3333", "33333",
				"11", "111", "333333", "3333333", "1111", "11111", "2222",
				"22222", "222222", "111111", "2222222", "1111111", "11111111",
				"111111111", "1111111111", "33333333", "333333333"
		};

		// Multitap EQ6 codes
		String[] mteq6code = {
				"4", "88", "777", "6", "111", "$", "#", "~", "[[",
				"^", "^^", "<", "*", "9", "]", "&", "1", "2", "5",
				"3", "<<", "8", "11", "77", "[", "7"
		};

		// Multitap 6-key codes
		String[] mt6code = {
				"1", "11", "111", "1111", "11111",
				"2", "22", "222", "2222", "22222",
				"3", "33", "333", "3333", "33333",
				"4", "44", "444", "4444", "44444",
				"5", "55", "555", "5555", "55555",
				"6"
		};

		// Multitap 4-key codes
		String[] mt4code = {
				"1", "11", "111", "1111", "11111", "111111", "1111111",
				"2", "22", "222", "2222", "22222", "222222", "2222222",
				"3", "33", "333", "3333", "33333", "333333", "3333333",
				"4", "44", "444", "4444", "44444"
		};

		// Multitap 2-key codes
		String[] mt2code = {
				"1", 
				"11", 
				"111", 
				"1111", 
				"11111", 
				"111111", 
				"1111111", 
				"11111111", 
				"111111111", 
				"1111111111", 
				"111111111111", 
				"1111111111111",
				"11111111111111",
				"2",
				"22",
				"222",
				"2222",
				"22222",
				"222222",
				"2222222",
				"22222222",
				"222222222",
				"2222222222",
				"22222222222",
				"222222222222",
				"2222222222222"
		};

		// Multitap 1-key codes
		String[] mt1code = {
				"1", 
				"11", 
				"111", 
				"1111", 
				"11111", 
				"111111", 
				"1111111", 
				"11111111", 
				"111111111", 
				"1111111111", 
				"11111111111", 
				"111111111111", 
				"1111111111111", 
				"11111111111111", 
				"111111111111111", 
				"1111111111111111", 
				"11111111111111111", 
				"111111111111111111", 
				"1111111111111111111", 
				"11111111111111111111", 
				"111111111111111111111", 
				"1111111111111111111111", 
				"11111111111111111111111", 
				"111111111111111111111111", 
				"1111111111111111111111111", 
				"11111111111111111111111111", 
		};

		// Multitap TouchMeKey4 codes
		String[] mttmk4code = {
				"1", "11", "111", "1111", "11111", "111111",
				"2", "22", "222", "2222", "22222", "222222",
				"3", "33", "333", "3333", "33333", "333333", "3333333",
				"4", "44", "444", "4444", "44444", "444444", "4444444"
		};

		// Multitap codes
		String[] MTcode = {
				"S",
				"2", "22", "222", "3", "33", "333", "4", "44", "444",
				"5", "55", "555", "6", "66", "666", "7", "77", "777", "7777",
				"8", "88", "888", "9", "99", "999", "9999",
				"2222", // Finnish (a with two dots)
				"6666"  // Finnish (o with two dots)
		};

		// Two thumb codes ('L' = left side, 'R' = right side)
		String[] twoThumbCode = {
				"L", "L", "L", "L", "L", "L", "L", "R", "R", "R", "R", "R",
				"R", "R", "R", "R", "L", "L", "L", "L", "R", "L", "L", "L",
				"R", "L"
		};

		// MessageEase codes 
		String[] MEcode = {
				"11", "56", "54", "58", "88", "95", "57", "44", "33", "59", "45",
				"25", "65", "22", "55", "53", "51", "53", "99", "77", "52",
				"15", "85", "35", "75", "89"
		};

		// Glenayre pager codes
		String[] Gcode = {
				"ULLLLS", "ULLLS", "ULLS", "ULS", "US",
				"URS", "URRS", "URRRS", "URRRRS", "URRRRRS",
				"LLLLS", "LLLS", "LLS", "LS",
				"RS", "RRS", "RRRS", "RRRRS",
				"DLS", "DS",
				"DRS", "DRRS", "DRRRS", "DRRRRS", "DRRRRRS", "DRRRRRRS"
		};

		// TiltWriter custom keyboard
		String[] tiltWriterCustom = {
				"abcd", "efgh", "ijklmn", "opqrst", "uvwxyz"    		  
		};

		//BufferedReader stdin =
		//		new BufferedReader(new InputStreamReader(System.in));

		String line;
		StringTokenizer st;
		//String meKeys = "";
		//int row = 0;
		//int col = 0;
		while ((line = inputFile.readLine()) != null)
		{
			st = new StringTokenizer(line);
			if (st.countTokens() != 2)
			{
				System.out.println("data format error!");
				System.exit(1);
			}
			String s1 = st.nextToken().toLowerCase();
			String freq = st.nextToken();

			String s2 = "";

			// *************************
			// TiltWriter custom keboard
			// *************************
			if (letterFreq && twc || digramFreq && twc)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}
			
			else if (wordFreq && twc)
			{
				String keystrokes = "";
				for (int i = 0; i < s1.length(); ++i)
					for (int j = 0; j < tiltWriterCustom.length; ++j)
					{
						//System.out.println(tiltWriterCustom[j] + " : " + s1.charAt(i));
						if (tiltWriterCustom[j].indexOf(s1.charAt(i)) >= 0)
						{
							keystrokes += "" + (j + 1);
							break;
						}
					}	
				System.out.println(s1 + " " + freq + " " + keystrokes);
			}

			//************************/
			//* Multitap 6-key input */
			//************************/

			//********************************
			else if (letterFreq && mt6 || digramFreq && mt6)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (wordFreq && mt6)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';

					// check if current key is same as the last key.  If so, insert
					// a press of "N" to segment the keypresses
					if (i > 0 && (mt6code[idx].charAt(0) == mt6code[s1.charAt(i - 1) - 'a'].charAt(0)))
						s2 += "N";					

					s2 += mt6code[idx];
				}
				s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//************************/
			//* Multitap 4-key input */
			//************************/

			//********************************
			else if (letterFreq && mt4 || digramFreq && mt4)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (wordFreq && mt4)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';

					// check if current key is same as the last key.  If so, insert
					// a press of "N" to segment the keypresses
					if (i > 0 && (mt4code[idx].charAt(0) == mt4code[s1.charAt(i - 1) - 'a'].charAt(0)))
						s2 += "N";						

					s2 += mt4code[idx];
				}
				s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//************************/
			//* Multitap 2-key input */
			//************************/

			//********************************
			else if (letterFreq && mt2 || digramFreq && mt2)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (wordFreq && mt2)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';

					// check if current key is same as the last key.  If so, insert
					// a press of "N" to segment the keypresses
					if (i > 0 && (mt2code[idx].charAt(0) == mt2code[s1.charAt(i - 1) - 'a'].charAt(0)))
						s2 += "N";					

					s2 += mt2code[idx];
				}
				s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//************************/
			//* Multitap 1-key input */
			//************************/

			//********************************
			else if (letterFreq && mt1 || digramFreq && mt1)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (wordFreq && mt1)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';

					// check if current key is same as the last key.  If so, insert
					// a press of "N" to segment the keypresses
					if (i > 0 && (mt1code[idx].charAt(0) == mt1code[s1.charAt(i - 1) - 'a'].charAt(0)))
						s2 += "N";					

					s2 += mt1code[idx];
				}
				s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//******************************/
			//* Multitap TouchMeKey4 input */
			//******************************/

			//********************************
			else if (letterFreq && mttmk4 || digramFreq && mttmk4)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (wordFreq && mttmk4)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';

					// check if current key is same as the last key.  If so, insert
					// a press of "N" to segment the keypresses
					if (i > 0 && (mttmk4code[idx].charAt(0) == mttmk4code[s1.charAt(i - 1) - 'a'].charAt(0)))
						s2 += "N";					

					s2 += mttmk4code[idx];
				}
				s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//************************/
			//* Multitap EQ6 input */
			//************************/

			//********************************
			else if (letterFreq && mteq6 || digramFreq && mteq6)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (wordFreq && mteq6)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';

					// check if current key is same as the last key.  If so, insert
					// a press of "N" to segment the keypresses
					if (i > 0 && (mteq6code[idx].charAt(0) == mteq6code[s1.charAt(i - 1) - 'a'].charAt(0)))
						s2 += "N";					

					s2 += mteq6code[idx];
				}
				s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//************************/
			//* Multitap EQ3 input */
			//************************/

			//********************************
			else if (letterFreq && mteq3 || digramFreq && mteq3)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (wordFreq && mteq3)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';

					// check if current key is same as the last key.  If so, insert
					// a press of "N" to segment the keypresses
					if (i > 0 && (mteq3code[idx].charAt(0) == mteq3code[s1.charAt(i - 1) - 'a'].charAt(0)))
						s2 += "N";					

					s2 += mteq3code[idx];
				}
				s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//************************/
			//* Multitap HL input    */
			//************************/

			//********************************
			else if (letterFreq && mthl || digramFreq && mthl)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (wordFreq && mthl)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';

					// check if current key is same as the last key.  If so, insert
					// a press of "N" to segment the keypresses
					if (i > 0 && (mthlcode[idx].charAt(0) == mthlcode[s1.charAt(i - 1) - 'a'].charAt(0)))
						s2 += "N";					

					s2 += mthlcode[idx];
				}
				s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//************************/
			//* Multitap LE input    */
			//************************/

			//********************************
			else if (letterFreq && mtle || digramFreq && mtle)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (wordFreq && mtle)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';

					// check if current key is same as the last key.  If so, insert
					// a press of "N" to segment the keypresses
					if (i > 0 && (mtlecode[idx].charAt(0) == mtlecode[s1.charAt(i - 1) - 'a'].charAt(0)))
						s2 += "N";					

					s2 += mtlecode[idx];
				}
				s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//**************************/
			//* Multitap FLPK input    */
			//**************************/

			//********************************
			else if (letterFreq && mtflpk || digramFreq && mtflpk)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (wordFreq && mtflpk)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';

					// check if current key is same as the last key.  If so, insert
					// a press of "N" to segment the keypresses
					if (i > 0 && (mtflpkcode[idx].charAt(0) == mtflpkcode[s1.charAt(i - 1) - 'a'].charAt(0)))
						s2 += "N";					

					s2 += mtflpkcode[idx];
				}
				s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//*****************************/
			//* Multitap LessTap input    */
			//*****************************/

			//********************************
			else if (letterFreq && mtlt || digramFreq && mtlt)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (wordFreq && mtlt)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';

					// check if current key is same as the last key.  If so, insert
					// a press of "N" to segment the keypresses
					if (i > 0 && (mtltcode[idx].charAt(0) == mtltcode[s1.charAt(i - 1) - 'a'].charAt(0)))
						s2 += "N";					

					s2 += mtltcode[idx];
				}
				s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//**************************/
			//* Multitap Stick input    */
			//**************************/

			//********************************
			else if (letterFreq && mtstick || digramFreq && mtstick)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (wordFreq && mtstick)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';

					// check if current key is same as the last key.  If so, insert
					// a press of "N" to segment the keypresses
					if (i > 0 && (mtstickcode[idx].charAt(0) == mtstickcode[s1.charAt(i - 1) - 'a'].charAt(0)))
						s2 += "N";					

					s2 += mtstickcode[idx];
				}
				s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//*******************************************************/
			//* Multitap Alphabetically-Constrained Design input    */
			//*******************************************************/

			//********************************
			else if (letterFreq && mtacd || digramFreq && mtacd)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (wordFreq && mtacd)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';

					// check if current key is same as the last key.  If so, insert
					// a press of "N" to segment the keypresses
					if (i > 0 && (mtacdcode[idx].charAt(0) == mtacdcode[s1.charAt(i - 1) - 'a'].charAt(0)))
						s2 += "N";					

					s2 += mtacdcode[idx];
				}
				s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//*********************************/
			//* Multitap AKKO Design input    */
			//*********************************/

			//********************************
			else if (letterFreq && mtakko || digramFreq && mtakko)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (wordFreq && mtakko)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';

					// check if current key is same as the last key.  If so, insert
					// a press of "N" to segment the keypresses
					if (i > 0 && (mtakkocode[idx].charAt(0) == mtakkocode[s1.charAt(i - 1) - 'a'].charAt(0)))
						s2 += "N";					

					s2 += mtakkocode[idx];
				}
				s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//************************/
			//* Multitap SureType input */
			//************************/

			//********************************
			else if (letterFreq && mtst || digramFreq && mtst)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (wordFreq && mtst)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';

					// check if current key is same as the last key.  If so, insert
					// a press of "N" to segment the keypresses
					if (i > 0 && (mtstcode[idx].charAt(0) == mtstcode[s1.charAt(i - 1) - 'a'].charAt(0)))
						s2 += "N";					

					s2 += mtstcode[idx];
				}
				s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			// =======================================
			// B A S I C   E N C O D I N G   M O D E S
			// =======================================

			//********************************************/
			//* Letters on 6/4/2/1 keys (basic encoding) */
			//********************************************/

			//********************************
			else if (letterFreq && k6 || digramFreq && k6  ||
					letterFreq && k4 || digramFreq && k4  ||
					letterFreq && k2 || digramFreq && k2  ||
					letterFreq && k1 || digramFreq && k1  ||
					letterFreq && sureType || digramFreq && sureType ||
					letterFreq && sureTypeModified || digramFreq && sureTypeModified ||
					letterFreq && eq3 || digramFreq && eq3 ||
					letterFreq && eq6 || digramFreq && eq6)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (wordFreq && k6)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';
					s2 += mt6code[idx].charAt(0);
				}
				//s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//********************************
			else if (wordFreq && k4)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';
					s2 += mt4code[idx].charAt(0);
				}
				//s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//********************************
			else if (wordFreq && k2)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';
					s2 += mt2code[idx].charAt(0);
				}
				//s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//********************************
			else if (wordFreq && k1)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';
					s2 += mt1code[idx].charAt(0);
				}
				//s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//********************************
			else if (wordFreq && tmk4)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';
					s2 += mttmk4code[idx].charAt(0);
				}
				//s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//********************************
			else if (wordFreq && sureType)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';
					s2 += mtstcode[idx].charAt(0);
				}
				//s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//********************************
			else if (wordFreq && sureTypeModified)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';
					s2 += mtstmcode[idx].charAt(0);
				}
				//s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//********************************
			else if (wordFreq && eq3)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';
					s2 += mteq3code[idx].charAt(0);
				}
				//s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//********************************
			else if (wordFreq && eq6)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';
					s2 += mteq6code[idx].charAt(0);
				}
				//s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//********************************
			else if (wordFreq && hl)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';
					s2 += mthlcode[idx].charAt(0);
				}
				//s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//********************************
			else if (wordFreq && le)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';
					s2 += mtlecode[idx].charAt(0);
				}
				//s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//********************************
			else if (wordFreq && flpk)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';
					s2 += mtflpkcode[idx].charAt(0);
				}
				//s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//********************************
			else if (wordFreq && lt)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';
					s2 += mtltcode[idx].charAt(0);
				}
				//s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//********************************
			else if (wordFreq && stick)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';
					s2 += mtstickcode[idx].charAt(0);
				}
				//s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//********************************
			else if (wordFreq && acd)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';
					s2 += mtacdcode[idx].charAt(0);
				}
				//s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//********************************
			else if (wordFreq && akko)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';
					s2 += mtakkocode[idx].charAt(0);
				}
				//s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//**************/
			//* MessagEase */
			//**************/

			//***************************
			else if (letterFreq && messagEase)
			{
				int idx = s1.charAt(0) - 'a';

				if (s1.equals("_"))
					s2 = "S";
				else
					s2 = MEcode[idx];
				System.out.println(s1 + " " + freq + " " + s2);
			}

			// *******************************
			else if (digramFreq && messagEase)
			{
				int idx2 = s1.charAt(1) - 'a';

				if (s1.substring(1, 2).equals("_"))
					s2 += "S";
				else
					s2 += MEcode[idx2];

				System.out.println(s1 + " " + freq + " " + s2);
			}

			// *******************************
			else if (wordFreq && messagEase)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';

					if (s1.substring(i, i + 1).equals("_"))
						s2 += "S";
					else
						s2 += MEcode[idx];
				}
				s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}


			//*********/
			//* Pager */
			//*********/

			//***************************
			else if (letterFreq && pager)
			{
				int idx = s1.charAt(0) - 'a';

				if (s1.equals("_"))
					s2 = "S";
				else
					s2 = Gcode[idx];

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//***************************
			else if (digramFreq && pager)
			{
				int idx = s1.charAt(1) - 'a';

				if (s1.substring(1, 2).equals("_"))
					s2 = "S";
				else
					s2 = Gcode[idx];

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//***************************
			else if (wordFreq && pager)
			{
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.charAt(i) - 'a';

					if (s1.substring(i, i + 1).equals("_"))
						s2 += "S";
					else
						s2 += Gcode[idx];
				}
				s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//************************/
			//* Date stamp method #1 */
			//************************/

			//********************************
			else if (letterFreq && dateStamp1)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (digramFreq && dateStamp1)
			{
				char c1 = s1.charAt(0);
				char c2 = s1.charAt(1);
				int idx1 = c1 == '_' ? 0 : c1 - 'a' + 1; 
				int idx2 = c2 == '_' ? 0 : c2 - 'a' + 1;

				int diff = Math.abs(idx1 - idx2);
				String key = idx1 < idx2 ? "R" : "L";

				for (int i = 0; i < diff; ++i)
					s2 += key;
				s2 += "S";

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//******************************
			else if (wordFreq && dateStamp1)
			{
				String temp = "_" + s1 + "_";

				for (int j = 1; j < temp.length(); ++j)
				{
					char c1 = temp.charAt(j - 1);
					char c2 = temp.charAt(j);

					int idx1 = c1 == '_' ? 0 : c1 - 'a' + 1; 
					int idx2 = c2 == '_' ? 0 : c2 - 'a' + 1;

					int diff = Math.abs(idx1 - idx2);
					String key = idx1 < idx2 ? "R" : "L";

					for (int i = 0; i < diff; ++i)
						s2 += key;
					s2 += "S";
				}

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//************************/
			//* Date stamp method #2 */
			//************************/

			//********************************
			else if (letterFreq && dateStamp2)
			{
				char c = s1.charAt(0);
				int idx = c == '_' ? 0 : c - 'a' + 1; 

				for (int i = 0; i < idx; ++i)
					s2 += "R";
				s2 += "S";
				System.out.println(s1 + " " + freq + " " + s2);
			}

			else if (digramFreq && dateStamp2)
			{
				char c = s1.charAt(1);
				int idx = c == '_' ? 0 : c - 'a' + 1;

				for (int i = 0; i < idx; ++i)
					s2 += "R";
				s2 += "S";
				System.out.println(s1 + " " + freq + " " + s2);
			}

			//******************************
			else if (wordFreq && dateStamp2)
			{
				String temp = s1 + "_";

				for (int j = 0; j < temp.length(); ++j)
				{
					char c = temp.charAt(j);

					int idx = c == '_' ? 0 : c - 'a' + 1; 

					for (int i = 0; i < idx; ++i)
						s2 += "R";
					s2 += "S";
				}
				System.out.println(s1 + " " + freq + " " + s2);
			}

			//************************/
			//* Date stamp method #3 */
			//************************/

			//********************************
			else if (letterFreq && dateStamp3)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (digramFreq && dateStamp3)
			{
				int idx1 = 0;
				int idx2 = 0;

				char c1 = s1.charAt(0);
				char c2 = s1.charAt(1);

				if (c1 == '_')
					idx1 = 13;
				else if (c1 >= 'a' && c1 <= 'm')
					idx1 = c1 - 'a';
				else if (c1 >= 'n' && c1 <= 'z')
					idx1 = c1 - 'n' + 14;
				else
				{
					System.out.println("data error!");
					System.exit(1);
				}

				if (c2 == '_')
					idx2 = 13;
				else if (c2 >= 'a' && c2 <= 'm')
					idx2 = c2 - 'a';
				else if (c2 >= 'n' && c2 <= 'z')
					idx2 = c2 - 'n' + 14;
				else
				{
					System.out.println("data error!");
					System.exit(1);
				}

				int diff = Math.abs(idx1 - idx2);
				String key = idx2 > idx1 ? "R" : "L";

				for (int j = 0; j < diff; ++j)
					s2 += key;

				s2 += "S";
				System.out.println(s1 + " " + freq + " " + s2);
			}

			//******************************
			else if (wordFreq && dateStamp3)
			{
				String temp = "_" + s1 + "_";

				for (int i = 1; i < temp.length(); ++i)
				{
					int idx1 = 0;
					int idx2 = 0;

					char c1 = temp.charAt(i - 1);
					char c2 = temp.charAt(i);

					if (c1 == '_')
						idx1 = 13;
					else if (c1 >= 'a' && c1 <= 'm')
						idx1 = c1 - 'a';
					else if (c1 >= 'n' && c1 <= 'z')
						idx1 = c1 - 'n' + 14;
					else
					{
						System.out.println("data error!");
						System.exit(1);
					}

					if (c2 == '_')
						idx2 = 13;
					else if (c2 >= 'a' && c2 <= 'm')
						idx2 = c2 - 'a';
					else if (c2 >= 'n' && c2 <= 'z')
						idx2 = c2 - 'n' + 14;
					else
					{
						System.out.println("data error!");
						System.exit(1);
					}

					int diff = Math.abs(idx1 - idx2);
					String key = idx2 > idx1 ? "R" : "L";

					for (int j = 0; j < diff; ++j)
						s2 += key;

					s2 += "S";
				}
				System.out.println(s1 + " " + freq + " " + s2);
			}

			//************************/
			//* Date stamp method #4 */
			//************************/

			//********************************
			else if (letterFreq && dateStamp4)
			{
				int idx = 0;
				char c = s1.charAt(0);

				if (c == '_')
					idx = 13;
				else if (c >= 'a' && c <= 'm')
					idx = c - 'a';
				else if (c >= 'n' && c <= 'z')
					idx = c - 'n' + 14;
				else
				{
					System.out.println("data error!");
					System.exit(1);
				}

				int diff = Math.abs(13 - idx);

				String key = idx > 13 ? "R" : "L";

				for (int j = 0; j < diff; ++j)
					s2 += key;

				s2 += "S";
				System.out.println(s1 + " " + freq + " " + s2);
			}

			//********************************
			else if (digramFreq && dateStamp4)
			{
				int idx = 0;
				char c = s1.charAt(1);

				if (c == '_')
					idx = 13;
				else if (c >= 'a' && c <= 'm')
					idx = c - 'a';
				else if (c >= 'n' && c <= 'z')
					idx = c - 'n' + 14;
				else
				{
					System.out.println("data error!");
					System.exit(1);
				}

				int diff = Math.abs(13 - idx);

				String key = idx > 13 ? "R" : "L";

				for (int j = 0; j < diff; ++j)
					s2 += key;

				s2 += "S";
				System.out.println(s1 + " " + freq + " " + s2);
			}

			//********************************
			else if (wordFreq && dateStamp4)
			{
				String temp = "_" + s1 + "_";

				for (int i = 1; i < temp.length(); ++i)
				{
					int idx = 0;
					char c = temp.charAt(i);

					if (c == '_')
						idx = 13;
					else if (c >= 'a' && c <= 'm')
						idx = c - 'a';
					else if (c >= 'n' && c <= 'z')
						idx = c - 'n' + 14;
					else
					{
						System.out.println("data error!");
						System.exit(1);
					}

					int diff = Math.abs(13 - idx);

					String key = idx > 13 ? "R" : "L";

					for (int j = 0; j < diff; ++j)
						s2 += key;

					s2 += "S";
				}
				System.out.println(s1 + " " + freq + " " + s2);
			}

			//************************/
			//* Date stamp method #7 */
			//************************/

			//********************************
			else if (letterFreq && dateStamp7)
			{
				String layout = "_etaoinshrldcufmwypgbvkxjqz";
				String keystrokes = "";

				for (int i = 0; i < layout.indexOf(s1); ++i)
					keystrokes += "R";
				keystrokes += "S"; 

				System.out.println(s1 + " " + freq + " " + keystrokes);
			}

			//********************************
			else if (digramFreq && dateStamp3)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (wordFreq && dateStamp3)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			} 

			//*****************/
			//* Qwerty method */
			//*****************/

			//********************************
			else if (letterFreq && qwerty)
				System.out.println(s1 + " " + freq + " " + s1 + "_");

			//********************************
			else if (digramFreq && qwerty)
				System.out.println(s1 + " " + freq + " " + s1.charAt(1) + "_");

			//********************************
			else if (wordFreq && qwerty)
				System.out.println(s1 + " " + freq + " " + s1 + "_");

			//*******************/
			//* Multitap method */
			//*******************/

			//********************************
			else if (letterFreq && multitap)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//********************************
			else if (digramFreq && multitap)
			{
				char c1 = s1.charAt(0);
				char c2 = s1.charAt(1);

				//int idx1 = c1 == '_' ? 26 : c1 - 'a';
				//int idx2 = c2 == '_' ? 26 : c2 - 'a';

				if (c2 == '_') s2 = "S";
				else
					//s2 = MTcode[idx2];
					s2 = MTcode[characters.indexOf(c2)];

				//if (MTcode[idx1].charAt(0) == MTcode[idx2].charAt(0))
				if (MTcode[characters.indexOf(c1)].charAt(0) ==
						MTcode[characters.indexOf(c2)].charAt(0))
					s2 = "N" + s2;

				System.out.println(s1 + " " + freq + " " + s2);
			}

			//********************************
			else if (wordFreq && multitap)
				System.out.println(s1 + " " + freq + " " +
						EncodedWord.encodeWordMultitap(s1) + "S");

			//*****************************/
			//* Multitap Optimized method */
			//*****************************/

			//********************************
			else if (letterFreq && multitapOptimized)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//*********************/
			//* LetterWise method */
			//*********************/

			//********************************
			else if (letterWise || t9)
			{
				System.out.println("Sorry, argument combination not supported!");
				System.exit(1);
			}

			//***************************/
			//* Telephone Keypad method */
			//***************************/

			//********************************
			else if (letterFreq && phoneKeypad)
				System.out.println(s1 + " " + freq + " " +
						EncodedWord.encodeWord(s1));

			//********************************
			else if (digramFreq && phoneKeypad)
				System.out.println(s1 + " " + freq + " " +
						EncodedWord.encodeWord(s1.substring(0, 1)));

			//********************************
			else if (wordFreq && phoneKeypad)
				System.out.println(s1 + " " + freq + " " +
						EncodedWord.encodeWord(s1));  // NOTE: no final SPACE

			//********************/
			//* Two Thumb Method */
			//********************/

			//****************************
			else if (wordFreq && twoThumb)
			{
				String twoThumbKeystrokes = "";
				for (int i = 0; i < s1.length(); ++i)
				{
					int idx = s1.toLowerCase().charAt(i) - 'a';
					twoThumbKeystrokes += twoThumbCode[idx];
				} 
				System.out.println(s1 + " " + freq + " " + twoThumbKeystrokes);
			}			

			//*******************/
			//* Wrong arguments */
			//*******************/
			else
			{
				System.out.println("Unknown arguments!");
				System.exit(1);
			}
		}
	}
}

