package ca.yorku.cse.mack.tilttarget;

import ca.yorku.cse.mack.tilttarget.Key;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.media.MediaPlayer;
import android.os.CountDownTimer;
import android.os.Vibrator;
import android.util.AttributeSet;
import android.view.Display;
import android.view.View;
import android.view.WindowManager;

/**
 * TiltTarget
 * 
 * @author Scott MacKenzie
 *
 */
public class ExperimentPanel extends View 
{
	final float DEGREES_TO_RADIANS = 0.0174532925f;
    final int DEFAULT_BALL_DIAMETER = 20;
    final int RESULTS_TEXT_SIZE = 40; // may have to tweak, depending on device
    
    int ballDiameter;
    float scaleFactor;
    int gutterWidth;
    
    final int NUMBER_OF_KEYS = 15;
    final int NUMBER_OF_COLUMNS = 3;
    final String KEY_LABELS = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";    
    final String FIVE_BY_THREE = "5x3";
    final String FOUR_BY_TEN = "4x10";
        
    String keyboard, orderOfControl;
    int numberOfLetterKeys, numberOfWordKeys;
    String[] keyLabel;
    int[] keyIndex;
    float velocity;
    int keyIndexToSelect;
    float dT;
    long now, lastT;
    
	Vibrator vib;
	MediaPlayer clickSound;
	Bitmap ball;	
    Key[] key; 
    Key tentativeKey, continueKey;
	int numberOfKeys, numberOfRows, numberOfColumns;
	float keyHeight, keyWidth;
    int panelWidth, panelHeight;
	boolean parametersAreSet = false;
	float xBall, yBall; // top-left of the ball (for painting)
	float xBallCenter, yBallCenter; // Centre of the ball
	boolean vibrotactileFeedback, auditoryFeedback;
	long dwellTime;
	boolean dwellPending;
	CountDownTimer dwellTimer;
	float tiltAngle, tiltMagnitude;
	int tiltGain; 
	float dBall; // the amount to move the ball (in pixels): dBall = dT * velocity	
	float xCenter, yCenter; // the centre of the screen
	Paint resultsBackgroundPaint, resultsTextPaint, normalPaint, fillPaint, fillHoverPaint, greenFillPaint, textPaint;
	int displayWidth, displayHeight;
	
	// NOTE: these variables are accessed in TiltTargetActivity
	boolean keySelected = false; // any key
	boolean targetSelected = false; // the key to select
	boolean endOfSequence = true; // don't allow input until enabled in main activity
	StringBuilder resultsString;
	boolean done = false; // ...but give the user a chance to view results from last phrase
	boolean reallyDone = false; // close files and exit
	//-----
		
	public ExperimentPanel(Context ctx)
	{
		super(ctx);
		initialize(ctx);
	}
	
	public ExperimentPanel(Context ctx, AttributeSet attrs) 
	{
		super(ctx, attrs);
		initialize(ctx);
	}
	
	public ExperimentPanel(Context ctx, AttributeSet attrs, int defStyle) 
	{
		super(ctx, attrs, defStyle);
		initialize(ctx);
	}
	
	// things that can be initialised from within this View
	private void initialize(Context ctx)
	{	
		ballDiameter = DEFAULT_BALL_DIAMETER;
		gutterWidth = 20;
		
		WindowManager wm = (WindowManager) ctx.getSystemService(Context.WINDOW_SERVICE);
		Display display = wm.getDefaultDisplay();
		displayWidth = display.getWidth();		
		displayHeight = display.getHeight();
		
		setKeyboard(FIVE_BY_THREE); // default
		configureKeys();
		numberOfKeys = NUMBER_OF_KEYS;
		numberOfColumns = NUMBER_OF_COLUMNS;
		        
        Bitmap temp = BitmapFactory.decodeResource(getResources(), R.drawable.ball);
        ball = Bitmap.createScaledBitmap(temp, ballDiameter, ballDiameter, true);
        
        numberOfKeys = NUMBER_OF_KEYS;
        key = new Key[numberOfKeys];
     
        resultsBackgroundPaint = new Paint();
        resultsBackgroundPaint.setColor(Color.BLACK);
        resultsBackgroundPaint.setStyle(Paint.Style.FILL);
		
        resultsTextPaint = new Paint();
        resultsTextPaint.setColor(Color.WHITE);
        resultsTextPaint.setAntiAlias(true);
        resultsTextPaint.setTextSize(RESULTS_TEXT_SIZE);
        
        resultsString = new StringBuilder(500);
        resultsString.append("WELCOME\n \nTo begin...\nPosition ball in\nGREEN square");
        
        greenFillPaint = new Paint();
        greenFillPaint.setColor(Color.GREEN);
        greenFillPaint.setStyle(Paint.Style.FILL);
		
        normalPaint = new Paint();
        normalPaint.setColor(0xffff9999); // lighter red (to minimize distraction)
        normalPaint.setStyle(Paint.Style.STROKE);
        normalPaint.setStrokeWidth(2);
        normalPaint.setAntiAlias(true);
        
        fillPaint = new Paint();
        fillPaint.setColor(0xffff9999); // lighter red (to minimize distraction)
        fillPaint.setStyle(Paint.Style.FILL);
        fillPaint.setStrokeWidth(2);
        fillPaint.setAntiAlias(true);
        
        fillHoverPaint = new Paint();
        fillHoverPaint.setColor(0xffff6666); // slight more red (to indicate ball-over)
        fillHoverPaint.setStyle(Paint.Style.FILL);
        fillHoverPaint.setStrokeWidth(2);
        fillHoverPaint.setAntiAlias(true);
        
        textPaint = new Paint();
        textPaint.setColor(Color.BLUE); 
        textPaint.setAntiAlias(true);
        textPaint.setTextAlign(Paint.Align.CENTER);
        textPaint.setTextSize(0.6f * keyHeight);
 
		this.setBackgroundColor(Color.LTGRAY);
	}
		 
	// things to initialise from the Activity
	public void setParameters(String keyboardArg, String orderOfControlArg,
			int tiltGainArg, int dwellTimeArg, 
			Vibrator vibArg, boolean vibrotactileFeedbackArg, 
			MediaPlayer clickSoundArg, boolean auditoryFeedbackArg)
	{
		setKeyboard(keyboardArg);
		configureKeys();
		orderOfControl = orderOfControlArg;
		tiltGain = tiltGainArg;
		dwellTime = dwellTimeArg;
		
		vib = vibArg;
		vibrotactileFeedback = vibrotactileFeedbackArg;
		clickSound = clickSoundArg;
		auditoryFeedback = auditoryFeedbackArg;
	
		xCenter = panelWidth / 2f;
		yCenter = panelHeight / 2f;	
		xBall = xCenter; // start the ball in the centre of the display
		yBall = yCenter;   
		
	   	dwellTimer  = new CountDownTimer(dwellTime, dwellTime) 
	   		{	
	   			public void onTick(long millisUntilFinished) { }	
	   			public void onFinish() 
	   			{ 
	   				dwellPending = false;	
	   				doTargetSelected();
	   			}
	   		};
		parametersAreSet = true;
	} // end setParameters
	
	private void configureKeys()
	{
		for (int i = 0; i < key.length; ++i)
		{
			float x = gutterWidth + (i % numberOfColumns) * keyWidth;
			float y = gutterWidth + (i / numberOfColumns) * keyHeight;
			key[i] = new Key(keyLabel[i], keyIndex[i]);
			key[i].setIdx(i);
			key[i].setCoordinates(x, y, keyWidth, keyHeight);
		}
		continueKey = new Key("", -1);
		continueKey.setIdx(0);
		continueKey.setCoordinates(panelWidth - ballDiameter, 0, ballDiameter + 1, ballDiameter + 1);
	}	
	
	public void setKeyboard(String s)
	{			
		if (s.equals(FIVE_BY_THREE)) // phone-like
		{
			panelWidth = displayWidth;
			panelHeight = displayHeight - 200; // MAGIC NUMBER! Ugh!
			numberOfRows = 5;
			numberOfColumns = 3;
			numberOfKeys = 15;
		} else if (s.equals(FOUR_BY_TEN)) // qwerty-like	
		{
			panelHeight = displayHeight - 100; 
			panelWidth = displayWidth; // MAGIC NUMBER! Ugh!
			numberOfRows = 4;
			numberOfColumns = 10;
			numberOfKeys = 40;
		}
		key = new Key[numberOfKeys];
		keyLabel = new String[numberOfKeys];
		keyIndex = new int[numberOfKeys];
		
		for (int i = 0; i < numberOfKeys; ++i)
		{
			keyLabel[i] = "" + KEY_LABELS.charAt(i);
			keyIndex[i] = i;
		}
		keyWidth = (panelWidth - 2 * gutterWidth) / numberOfColumns;
		keyHeight = (panelHeight - 2 * gutterWidth) / numberOfRows;
		
		// adjust ball diameter to be 25% of key width or height (whichever is less)
		ballDiameter = keyWidth < keyHeight ? (int)(0.25f * keyWidth) : (int)(0.25f * keyHeight);
        Bitmap temp = BitmapFactory.decodeResource(getResources(), R.drawable.ball);
        ball = Bitmap.createScaledBitmap(temp, ballDiameter, ballDiameter, true); // (re)adjust ball diameter in bitmap
	}
	
	// used from TiltWriterActivity
	public void updateTilt(float tiltAngleArg, float tiltMagnitudeArg) 
	{
		tiltAngle = tiltAngleArg;
		tiltMagnitude = tiltMagnitudeArg;
		
		// don't allow tiltMagnitude to exceed 45 degrees
		final float MAX_MAGNITUDE = 45f; 
		tiltMagnitude = tiltMagnitude > MAX_MAGNITUDE ? MAX_MAGNITUDE : tiltMagnitude;
	}

	protected void onDraw(Canvas canvas) 
	{		
		if (!parametersAreSet) 
			return;
	
		// get current time and delta since last onDraw
		now = System.nanoTime();
		dT = (now - lastT) / 1000000000f; // seconds
        lastT = now;	
        
		// This is the only code that distinguishes velocity-control from position-control
		if (orderOfControl.equals("Velocity")) // velocity control
		{
			// compute how far the ball should move
			velocity = tiltMagnitude * tiltGain;
			dBall = dT * velocity; // make the ball move this amount (pixels)
			
			// compute the ball's new coordinates
			float dx = (float)Math.sin(tiltAngle * DEGREES_TO_RADIANS) * dBall;
			float dy = -(float)Math.cos(tiltAngle * DEGREES_TO_RADIANS) * dBall;
			xBall += dx;
			yBall += dy;
		}
		else // position control
		{
			// compute how far the ball should move
			dBall = tiltMagnitude * tiltGain; 
			
			// compute the ball's new coordinates
			float dx = (float)Math.sin(tiltAngle * DEGREES_TO_RADIANS) * dBall;
			float dy = -(float)Math.cos(tiltAngle * DEGREES_TO_RADIANS) * dBall;
			xBall = xCenter + dx;
			yBall = yCenter + dy;
		}
		
		// keep the ball visible
		if (xBall < 0) xBall = 0;
		else if (xBall > panelWidth - ballDiameter) xBall = panelWidth - ballDiameter;
		if (yBall < 0) yBall = 0;
		else if (yBall > panelHeight - ballDiameter) yBall = panelHeight - ballDiameter;
		
		// adjust the ball's centre coordinates
		xBallCenter = xBall + ballDiameter / 2f;
		yBallCenter = yBall + ballDiameter / 2f;
		
		// take one of two draw paths, depending on whether we are "within a sequence" or "between sequences"
		if (endOfSequence)
		//if (false)
		{
			// ========================================
			// end of sequence: draw the results string
			// ========================================
			
			// draw the rectangle within which the results are presented
			canvas.drawRect(gutterWidth, gutterWidth, panelWidth - gutterWidth, panelHeight - gutterWidth, resultsBackgroundPaint);

			// x and y offsets for beginning of results string
			int xOffset = gutterWidth + 10;
			int yOffset = gutterWidth + 10 + RESULTS_TEXT_SIZE;
	
			// break results string by lines and draw the text
			//StringTokenizer st = new StringTokenizer(resultsString.toString(), "\n");
			//while (st.hasMoreTokens())
			//{
			//	canvas.drawText(st.nextToken(), xOffset, yOffset, resultsTextPaint);
			//	yOffset += resultsTextPaint.getTextSize();;
			//}
			
			String[] s = resultsString.toString().split("\n");
			for (int i = 0; i < s.length; ++i)
			{
				canvas.drawText(s[i], xOffset, yOffset, resultsTextPaint);
				yOffset += resultsTextPaint.getTextSize() + 4;				
			}
			
			// draw the green continue key
			canvas.drawRect(continueKey.x, continueKey.y, continueKey.x + continueKey.width, continueKey.y + continueKey.height, greenFillPaint);
			
			// if the ball enters the green key, start the dwell timer
			if (continueKey.inKey(xBallCenter, yBallCenter, ballDiameter))
			{
 	   			if (!dwellPending)
 	   			{
 	   				dwellPending = true;
 	   				dwellTimer.start();
 	   			}
			}
			
	        // cancel the dwell timer if the ball exits the continueKey (before the dwell time elapses)
	        if (dwellPending && !continueKey.inKey(xBallCenter, yBallCenter, ballDiameter)) 
	        {
	        	dwellPending = false;
	        	dwellTimer.cancel();     
	        }	
	        
		} else
		{
			// =====================================
			// within sequence: draw the target keys
			// =====================================			

			// configure the keys (set x-y coordinates, etc.)
	        configureKeys(); 
	        
	        // draw the keys
			for (int i = 0; i < key.length; ++i)
			{
				// draw the keys
				canvas.drawRect(key[i].x, key[i].y, key[i].x + key[i].width, key[i].y + key[i].height, normalPaint);

				// if it's the target key, fill it
				if (i == keyIndexToSelect)
					canvas.drawRect(key[i].x, key[i].y, key[i].x + key[i].width, key[i].y + key[i].height, fillPaint);				
				
				// draw the key text 
				canvas.drawText(key[i].label, key[i].r.centerX(), key[i].r.centerY() - textPaint.getFontMetrics().ascent / 2.0f, textPaint);
			}		
	      
			// determine which key, if any, the ball is inside
	        for (int i = 0; i < key.length; ++i)
	        {
	        	if (endOfSequence)
	        		break;
	        	if (key[i].inKey(xBallCenter, yBallCenter, ballDiameter))
	        	{
	        		// draw the target key text
	    			//canvas.drawText(key[i].label, key[i].r.centerX(), key[i].r.centerY() + textPaint.getTextSize() / 2.0f, textPaint);
	        		
	        		// if the ball is inside the target key, brighten the shading a bit
	        		if (i == keyIndexToSelect)
	        		{
	        			canvas.drawRect(key[i].x, key[i].y, key[i].x + key[i].width, key[i].y + key[i].height, fillHoverPaint);
	        			canvas.drawText(key[i].label, key[i].r.centerX(), key[i].r.centerY() - textPaint.getFontMetrics().ascent / 2.0f, textPaint);
	        		}

	    			// this is (tentatively) the key the user is trying to select (NOTE: dwell time must elapse before selection)
	        		tentativeKey = key[i];        		
	
	        		// start the dwell timer
	 	   			if (!dwellPending && !endOfSequence)
	 	   			{
	 	   				dwellPending = true;
	 	   				dwellTimer.start();
	 	   			}
	     	   		break;
	        	}
	        }
	        
	        // cancel the dwell timer if the ball exits the target key (before the dwell time elapses)
	        if (dwellPending && !key[tentativeKey.index].inKey(xBallCenter, yBallCenter, ballDiameter)) 
	        {
	        	dwellPending = false;
	        	dwellTimer.cancel();     
	        }	
        }
        
		// draw the ball in its new location
        canvas.drawBitmap(ball, xBall, yBall, null);             
	} // end onDraw

	private void doTargetSelected()
	{
		// provide feedback (as per setup)
		if (vibrotactileFeedback) vib.vibrate(5); 
		if (auditoryFeedback) clickSound.start();
		
		if (!endOfSequence)
		{
			// set keySelected
			keySelected = true; // used in TiltTargetActivity
			
			// set targetSelected as well, provided the ball is in the target
			if (tentativeKey.index == keyIndexToSelect)
				targetSelected = true; // used in TiltTargetActivity
		}
		else
			endOfSequence = false;		
		
		if (done)
			reallyDone = true;
		
		dwellTimer.cancel();
	}
	
	protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) 
	{
		setMeasuredDimension(panelWidth, panelHeight);
	}
}
