package ca.yorku.cse.mack.tilttarget;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Locale;
import java.util.Random;
import java.util.Vector;
import android.app.Activity;
import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.ActivityInfo;
import android.content.res.Configuration;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.media.MediaPlayer;
import android.media.MediaScannerConnection;
import android.net.Uri;
import android.os.Bundle;
import android.os.CountDownTimer;
import android.os.Environment;
import android.os.Vibrator;
import android.util.Log;
import android.view.Display;
import android.view.Surface;
import android.view.WindowManager;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.TextView;

/**
 * TiltTarget - Experiment software to evaluate/demonstrate tilt as a target selection method.
 * <p>
 * 
 * @author Scott MacKenzie
 * 
 */
public class TiltTargetActivity extends Activity implements SensorEventListener
{
	final static String MYDEBUG = "MYDEBUG"; // for Log.i messages
	
	final String WORKING_DIRECTORY = "/TiltTargetData/";
	
	final static int REFRESH_INTERVAL = 20; // milliseconds (screen refreshes @ 50 Hz)
	final String SD2_HEADER = "Participant,Session,Block,Group,Condition,"
			+ "Keyboard,Tilt_gain,Dwell_time,"
			+ "Desired_sequence,Actual_sequence,Time(s),Hits,Misses,Error_rate(%),Actual_ball_distance,Minimum_ball_distance,"
			+ "Movement_efficiency\n";
	final String FIVE_BY_THREE = "5x3";
	final String FOUR_BY_TEN = "4x10";
	final float RADIANS_TO_DEGREES = 57.2957795f;
	
	// int constants to setup a mode (see DemoTiltMeter API for discussion)
	final static int ORIENTATION = 0;
	final static int ACCELEROMETER_ONLY = 1;
	final static int ACCELEROMETER_AND_MAGNETIC_FIELD = 2;

	private SensorManager sm;
	private Sensor sO, sA, sM;
	int sensorMode;
	int defaultOrientation;
	
	TextView sequenceEnteredField, progressTextField;
	LinearLayout topPanel;
	int gapAtTop;
	ExperimentPanel ep;
	String participantCode, sessionCode, blockCode, groupCode, conditionCode;
	int sequencesPerBlock, targetsPerSequence;
	int sequenceCount;
	int keyToSelect, lastKeyToSelect;
	int numberOfKeys;
	String keyboard;
	String sequence;
	StringBuilder sequenceEntered, transcribedText, wordStem, sb1, sb2, resultsString, sd1Stuff, sd2Stuff;
	StringBuilder desiredSequence;
	Random r = new Random();
	Vector<Sample> samples;
	Vector<Point> actualPath, minimumPath;
	long t1, t2;
	int selectionCount; // the number of selections (whether or not the selection was on the target)
	int targetsSelected; // the number of targets selected (will increase up to targetsPerSequence)
	String orderOfControl;
	int tiltGain;
	int dwellTime;
	int samplingRate;
	boolean vibrotactileFeedback, auditoryFeedback;
	BufferedWriter sd1, sd2; // summary data 1 and summary data 2 files for output data
	File f1, f2;
	String sd1FileName, sd2FileName;
	Display display;
	ScreenRefreshTimer refreshScreen;

	int SMOOTH_OVER = 10; // number of samples to average over for smoothing
	int smoothIndex = 0;
	float[] xSmooth = new float[SMOOTH_OVER];
	float[] ySmooth = new float[SMOOTH_OVER];
	float[] zSmooth = new float[SMOOTH_OVER];
	boolean SMOOTHING_ENABLED = true;

	float[] orientation = new float[3];
	float[] accValues = new float[3];
	float[] magValues = new float[3];
	float x, y, z, pitch, roll;

	/**
	 * Below are the alpha values for the low-pass filter. The four values in each array are for the
	 * slowest to fastest sampling rates, respectively. These values were determined by trial and
	 * error. There is a trade-off. Generally, lower values produce smooth but sluggish responses,
	 * while higher values produced jerky but fast responses. There is also a difference by order of
	 * control; hence the use of two arrays. Furthermore, there appears to be difference by device
	 * (e.g., Samsung Galaxy Tab 10.1 vs. HTC Desire C). More work is needed here.
	 */
	// final float[] ALPHA_VELOCITY = { 0.99f, 0.8f, 0.4f, 0.15f };
	// final float[] ALPHA_POSITION = { 0.5f, 0.3f, 0.15f, 0.10f };
	float alpha;

	/** Called when the activity is first created. */
	@Override
	public void onCreate(Bundle savedInstanceState)
	{
		super.onCreate(savedInstanceState);
		setContentView(R.layout.main);
		findAllViewsById();

		display = getWindowManager().getDefaultDisplay();
		defaultOrientation = getDefaultDeviceOrientation();

		// init study parameters
		Bundle b = getIntent().getExtras();
		participantCode = b.getString("participantCode");
		sessionCode = b.getString("sessionCode");
		// blockCode = b.getString("blockCode");
		groupCode = b.getString("groupCode");
		conditionCode = b.getString("conditionCode");
		sequencesPerBlock = b.getInt("sequencesPerBlock");
		targetsPerSequence = b.getInt("targetsPerSequence");
		keyboard = b.getString("keyboard");
		orderOfControl = b.getString("orderOfControl");
		tiltGain = b.getInt("tiltGain");
		dwellTime = b.getInt("dwellTime");
		samplingRate = b.getInt("samplingRate");
		vibrotactileFeedback = b.getBoolean("vibrotactileFeedback");
		auditoryFeedback = b.getBoolean("auditoryFeedback");

		if (keyboard.equals(FIVE_BY_THREE))
		{
			numberOfKeys = 15;
			setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_PORTRAIT);
		} else if (keyboard.equals(FOUR_BY_TEN))
		{
			numberOfKeys = 40;
			setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE);
		}

		sequenceEntered = new StringBuilder(100);
		desiredSequence = new StringBuilder(100);
		transcribedText = new StringBuilder(100);
		progressTextField.setCursorVisible(false);
		updateProgressText(targetsPerSequence, 0, 0);
		samples = new Vector<Sample>();
		actualPath = new Vector<Point>();
		minimumPath = new Vector<Point>();
		wordStem = new StringBuilder(100);
		selectionCount = 0;
		resultsString = new StringBuilder(500);
		targetsSelected = 0;
		sequenceCount = 0;
		sequence = "";
		lastKeyToSelect = -1;

		// ===================
		// File initialization
		// ===================

		// make a working directory to store sd1 and sd2 data files
		File dataDirectory = new File(Environment.getExternalStorageDirectory() + WORKING_DIRECTORY);
		if (!dataDirectory.exists() && !dataDirectory.mkdirs())
		{
			Log.i("MYDEBUG", "Failed to create directory: " + WORKING_DIRECTORY);
			super.onDestroy(); // cleanup
			this.finish(); // terminate
		}
		Log.i("MYDEBUG", "dataDirectory = " + dataDirectory);

		// initialize output files
		int blockNumber = 1;
		blockCode = "B01";
		String base = "TiltTarget-" + participantCode + "-" + keyboard + "-" + sessionCode + "-" + blockCode + "-"
				+ groupCode + "-" + conditionCode + "-" + orderOfControl.substring(0, 3).toUpperCase() + "-" + "TG"
				+ tiltGain + "-" + "DT" + dwellTime;
		f1 = new File(dataDirectory, base + ".sd1");
		f2 = new File(dataDirectory, base + ".sd2");

		// make sure block code is unique (if not, increment block code and try again)
		while (f1.exists() || f2.exists())
		{
			Log.i("MYDEBUG", "f1 or f2 exists! blockCode=" + blockCode + ", f1.length=" + f1.length() + ", f2.length="
					+ f2.length());
			++blockNumber;
			blockCode = blockNumber < 10 ? "B0" + blockNumber : "B" + blockNumber;
			base = "TiltTarget-" + participantCode + "-" + keyboard + "-" + sessionCode + "-" + blockCode + "-"
					+ groupCode + "-" + conditionCode + "-" + orderOfControl.substring(0, 3).toUpperCase() + "-" + "TG"
					+ tiltGain + "-" + "DT" + dwellTime;

			f1 = new File(dataDirectory, base + ".sd1");
			f2 = new File(dataDirectory, base + ".sd2");
		}

		try
		{
			sd1 = new BufferedWriter(new FileWriter(f1));
			sd2 = new BufferedWriter(new FileWriter(f2));

			// output header in sd2 file
			sd2.write(SD2_HEADER, 0, SD2_HEADER.length());
			sd2.flush();
		} catch (IOException e)
		{
			Log.i("MYDEBUG", "Error opening data files! Exception: " + e.toString());
			super.onDestroy();
			this.finish();
		}
		// *** end file initialization ***

		ep.setParameters(keyboard, orderOfControl, tiltGain, dwellTime,
				(Vibrator)getSystemService(Context.VIBRATOR_SERVICE), vibrotactileFeedback, MediaPlayer.create(this,
						R.raw.click), auditoryFeedback);

		// ensure the target is not the same as the last target
		do
		{
			keyToSelect = r.nextInt(numberOfKeys);
		} while (keyToSelect == lastKeyToSelect);
		lastKeyToSelect = keyToSelect;

		ep.keyIndexToSelect = keyToSelect;
		desiredSequence.append(ep.KEY_LABELS.charAt(keyToSelect));
		minimumPath.addElement(new Point((int)(ep.key[keyToSelect].x), (int)(ep.key[keyToSelect].y)));
		getWindow().setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_STATE_ALWAYS_HIDDEN); 
		alpha = 0.1f; // constant for low-pass filter (might need to fiddle with this)
		
		// get sensors
		sm = (SensorManager)getSystemService(SENSOR_SERVICE);
		sO = sm.getDefaultSensor(Sensor.TYPE_ORIENTATION); // supported on many devices
		sA = sm.getDefaultSensor(Sensor.TYPE_ACCELEROMETER); // supported on most devices
		sM = sm.getDefaultSensor(Sensor.TYPE_MAGNETIC_FIELD); // null on many devices

		// setup the sensor mode (see API for discussion)
		if (sO != null)
		{
			sensorMode = ORIENTATION;
			sA = null;
			sM = null;
			Log.i(MYDEBUG, "Sensor mode: ORIENTATION");
		} else if (sA != null && sM != null)
		{
			sensorMode = ACCELEROMETER_AND_MAGNETIC_FIELD;
			Log.i(MYDEBUG, "Sensor mode: ACCELEROMETER_AND_MAGNETIC_FIELD");
		} else if (sA != null)
		{
			sensorMode = ACCELEROMETER_ONLY;
			Log.i(MYDEBUG, "Sensor mode: ACCELEROMETER_ONLY");
		} else
		{
			Log.i(MYDEBUG, "Can't run demo.  Requires Orientation sensor or Accelerometer");
			this.finish();
		}

		// NOTE: sensor listeners are registered in onResume
		
		// setup the screen refresh timer (updates every REFRESH_INTERVAL milliseconds)
		refreshScreen = new ScreenRefreshTimer(REFRESH_INTERVAL, REFRESH_INTERVAL);
		refreshScreen.start();
	}

	private void findAllViewsById()
	{
		sequenceEnteredField = (EditText)findViewById(R.id.presentedTry);
		progressTextField = (EditText)findViewById(R.id.transcribedTry);
		ep = (ExperimentPanel)findViewById(R.id.experimentpanel);
	}

	protected void onResume()
	{
		super.onResume();
		if (sO != null)
			sm.registerListener(this, sO, samplingRate);
		else
		{
			sm.registerListener(this, sA, samplingRate);
			sm.registerListener(this, sM, samplingRate);
		}
	}

	protected void onPause()
	{
		super.onPause();
		sm.unregisterListener(this);
	}

	float lastAccel[] = new float[3];
	float accelFilter[] = new float[3];
	float accel[] = new float[3];

	// implement SensorEventListener methods
	public void onAccuracyChanged(Sensor sensor, int accuracy)
	{
	}

	public void onSensorChanged(SensorEvent se)
	{
		if (ep.reallyDone)
			doDone();

		// ===============================
		// DETERMINE DEVICE PITCH AND ROLL
		// ===============================

		switch (sensorMode)
		{
			case ORIENTATION: // ========================================================

				/*
				 * Use this mode if the device has an orientation sensor.
				 */

				if (se.sensor.getType() != Sensor.TYPE_ORIENTATION)
				{
					Log.i(MYDEBUG, "Return now. Sensor event from " + se.sensor.getName());
					return;
				}

				// This bit of fiddling is necessary so the app will work on different devices.
				switch (defaultOrientation)
				{
					case Configuration.ORIENTATION_PORTRAIT:
					{
						// e.g., Nexus 4
						pitch = se.values[1];
						roll = se.values[2];
						break;
					}
					case Configuration.ORIENTATION_LANDSCAPE:
					{
						// e.g., Samsung Galaxy Tab 10.1
						pitch = se.values[2];
						roll = -se.values[1];
						break;
					}
				}
				break;

			case ACCELEROMETER_AND_MAGNETIC_FIELD: // ===================================

				/*
				 * Use this mode if the device has both an accelerometer and a magnetic field sensor
				 * (but no orientation sensor). See...
				 * 
				 * http://blog.thomnichols.org/2012/06/smoothing-sensor-data-part-2
				 */

				// smooth the sensor values using a low-pass filter
				if (se.sensor.getType() == Sensor.TYPE_ACCELEROMETER)
					accValues = lowPass(se.values.clone(), accValues, alpha);
				if (se.sensor.getType() == Sensor.TYPE_MAGNETIC_FIELD)
					magValues = lowPass(se.values.clone(), magValues, alpha);

				if (accValues != null && magValues != null)
				{
					// compute pitch and roll
					float R[] = new float[9];
					float I[] = new float[9];
					boolean success = SensorManager.getRotationMatrix(R, I, accValues, magValues);
					if (success) // see SensorManager API
					{
						float[] orientation = new float[3];
						SensorManager.getOrientation(R, orientation); // see getOrientation API
						pitch = orientation[1] * RADIANS_TO_DEGREES;
						roll = -orientation[2] * RADIANS_TO_DEGREES;
					}
				}
				break;

			case ACCELEROMETER_ONLY: // =================================================

				/*
				 * Use this mode if the device has an accelerometer but no magnetic field sensor and
				 * no orientation sensor (e.g., HTC Desire C, Asus MeMOPad). This algorithm doesn't
				 * work quite as well, unfortunately. See...
				 * 
				 * http://www.hobbytronics.co.uk/accelerometer-info
				 */

				// smooth the sensor values using a low-pass filter
				if (se.sensor.getType() == Sensor.TYPE_ACCELEROMETER)
					accValues = lowPass(se.values.clone(), accValues, alpha);

				x = accValues[0];
				y = accValues[1];
				z = accValues[2];
				pitch = (float)Math.atan(y / Math.sqrt(x * x + z * z)) * RADIANS_TO_DEGREES;
				roll = (float)Math.atan(x / Math.sqrt(y * y + z * z)) * RADIANS_TO_DEGREES;
				break;
		}
		
		if (keyboard.equals(FOUR_BY_TEN))
		{
			float temp = pitch;
			pitch = -roll;
			roll = temp;
		}
		

	}

	/*
	 * Get the default orientation of the device. This is needed to correctly map the sensor data
	 * for pitch and roll (see onSensorChanged). See...
	 * 
	 * http://stackoverflow.com/questions/4553650/how-to-check-device-natural-default-orientation-on-
	 * android-i-e-get-landscape
	 */
	public int getDefaultDeviceOrientation()
	{
		WindowManager windowManager = (WindowManager)getSystemService(WINDOW_SERVICE);
		Configuration config = getResources().getConfiguration();
		int rotation = windowManager.getDefaultDisplay().getRotation();

		if (((rotation == Surface.ROTATION_0 || rotation == Surface.ROTATION_180) && config.orientation == Configuration.ORIENTATION_LANDSCAPE)
				|| ((rotation == Surface.ROTATION_90 || rotation == Surface.ROTATION_270) && config.orientation == Configuration.ORIENTATION_PORTRAIT))
			return Configuration.ORIENTATION_LANDSCAPE;
		else
			return Configuration.ORIENTATION_PORTRAIT;
	}

	public void doEndOfSequence()
	{
		ep.endOfSequence = true;

		// compute distance the ball moved
		float distanceMoved = 0.0f;
		for (int i = 1; i < actualPath.size(); ++i)
		{
			int x1 = ((Point)actualPath.elementAt(i - 1)).x;
			int y1 = ((Point)actualPath.elementAt(i - 1)).y;
			int x2 = ((Point)actualPath.elementAt(i)).x;
			int y2 = ((Point)actualPath.elementAt(i)).y;
			int dx = x2 - x1;
			int dy = y2 - y1;
			distanceMoved += (float)Math.sqrt(dx * dx + dy * dy);
		}

		// compute the minimum possible distance the ball could have moved
		// NOTE: calculation uses the middle of the targets. Actual minimum
		// may be less if the ball is consistently moved only to the nearest edge of the targets.
		float minimumDistance = 0.0f;
		for (int i = 1; i < minimumPath.size(); ++i)
		{
			int x1 = ((Point)minimumPath.elementAt(i - 1)).x;
			int y1 = ((Point)minimumPath.elementAt(i - 1)).y;
			int x2 = ((Point)minimumPath.elementAt(i)).x;
			int y2 = ((Point)minimumPath.elementAt(i)).y;
			int dx = x2 - x1;
			int dy = y2 - y1;
			minimumDistance += (float)Math.sqrt(dx * dx + dy * dy);
		}

		// =========================
		// Results presented to user
		// =========================

		resultsString.delete(0, resultsString.length());
		resultsString.append("Thank you!\n \n");

		resultsString.append("Block " + Integer.parseInt(blockCode.substring(1)) + "\n");

		resultsString.append("Sequence " + (sequenceCount + 1) + " of " + sequencesPerBlock + "\n");

		float totalTime = trim(t2 / 1000.0f, 2);
		resultsString.append("Total time: " + totalTime + " s\n");

		resultsString.append("Hits: " + targetsSelected + "\n");
		resultsString.append("Misses: " + (selectionCount - targetsSelected) + "\n");

		String errorRate = "" + trim((selectionCount - targetsSelected) / targetsSelected * 100.0f, 2);
		resultsString.append("Error rate: " + errorRate + "%\n");

		resultsString.append("Ball distance: " + (int)distanceMoved + " px\n");
		resultsString.append("Min distance: " + (int)minimumDistance + " px\n");

		// NOTE: Movement efficiency can exceed 100% if movement is direct AND selection occurs at
		// the nearest edge of
		// each target
		float movementEfficiency = minimumDistance / distanceMoved * 100.0f;
		resultsString.append("Movement efficiency: " + trim(movementEfficiency, 1) + "%\n \n");

		resultsString.append("GREEN to continue");
		ep.resultsString = resultsString;

		// ==================================
		// Results saved to output data files
		// ==================================

		// sd2 stuff first

		sd2Stuff = new StringBuilder(500);
		sd2Stuff.append(participantCode + "," + // participant code
				sessionCode + "," + // session code
				blockCode + "," + // block code
				groupCode + "," + // group code
				conditionCode + "," + // condition code
				keyboard + "," + // keyboard
				tiltGain + "," + // tilt gain
				dwellTime + "," + // dwell time
				desiredSequence + "," + // desired sequence
				sequenceEntered + "," + // actual sequence entered
				totalTime + "," + // total time
				targetsSelected + "," + // hits
				(selectionCount - targetsSelected) + "," + // misses
				errorRate + "," + // error rate
				(int)distanceMoved + "," + // distance the ball moved (pixels)
				(int)minimumDistance + "," + // minimum distance the ball could have moved
				trim(movementEfficiency, 1) + "\n");

		// sd1 stuff next

		sd1Stuff = new StringBuilder(100);
		sd1Stuff.append(desiredSequence + "\n");
		sd1Stuff.append(sequenceEntered + "\n");
		for (int i = 0; i < samples.size(); ++i)
			sd1Stuff.append((Sample)samples.elementAt(i) + "\n");
		sd1Stuff.append("-----\n");

		// write to data files
		try
		{
			sd1.write(sd1Stuff.toString(), 0, sd1Stuff.length());
			sd1.flush();
			sd2.write(sd2Stuff.toString(), 0, sd2Stuff.length());
			sd2.flush();
		} catch (IOException e)
		{
			System.err.println("ERROR WRITING TO DATA FILE!\n" + e);
			System.exit(1);
		}

		// check if last sequence in block
		++sequenceCount;
		if (sequenceCount == sequencesPerBlock)
			ep.done = true; // ... but display results from last sequence before quitting
		else
			doNewSequence();
		return;
	}

	public void doNewSequence()
	{
		sequenceEntered.replace(0, sequenceEntered.length(), "");
		sequenceEnteredField.setText(sequenceEntered);
		desiredSequence.replace(0, desiredSequence.length(), "");
		updateProgressText(targetsPerSequence, 0, 0);
		t1 = 0;
		selectionCount = 0;
		targetsSelected = 0;
		samples = new Vector<Sample>();
		actualPath = new Vector<Point>();
		minimumPath = new Vector<Point>();
		return;
	}

	public void updateProgressText(int sequencesPerBlockArg, int targetsSelectedArg, int selectionCountArg)
	{
		int sequencesPerBlock = sequencesPerBlockArg;
		int targetsSelected = targetsSelectedArg;
		int selectionCount = selectionCountArg;

		StringBuilder s = new StringBuilder("Targets:     Hits:     Misses:       ");
		// 012345678901234567890123456789012345678901234567890123
		// 1 2 3 4 5
		// indices determined from the ruler above
		s.replace(9, 11, "" + sequencesPerBlock);
		s.replace(19, 20, "" + targetsSelected);
		s.replace(31, 33, "" + (selectionCount - targetsSelected));
		progressTextField.setText(s.toString());
	}

	// Done! close data files and exit
	private void doDone()
	{
		try
		{
			sd1.close();
			sd2.close();

			// Make the saved data files visible in Windows Explorer
			// There seems to be bug doing this with Android 4.4. I'm using the following
			// code, instead of sendBroadcast. See...
			// http://code.google.com/p/android/issues/detail?id=38282
			MediaScannerConnection.scanFile(this, new String[] { f1.getAbsolutePath(), f2.getAbsolutePath() }, null,
					null);
		} catch (IOException e)
		{
			// work on later (but no console to write to)
		}
		finish();
	}

	// ========================================================
	// Sample - simple class to hold a time stamp and keystroke
	// ========================================================

	private class Sample
	{
		private long time;
		private String key;

		Sample(long timeArg, String keyArg)
		{
			time = timeArg;
			key = keyArg;
		}

		public String toString()
		{
			return time + ", " + key;
		}
	}

	// return the mean of the values in array f of size n
	// modified to do triangular smoothing
	public float smooth(float[] f, int idx)
	{
		float average = 0.0f;
		// int factor = 0;
		for (int i = 0; i < f.length; ++i)
		{
			// int temp = f.length - idx + i;
			// if (temp > f.length) temp = temp % f.length;
			// average += f[i] * temp;
			average += f[i];
			// factor += temp;
			// Log.i("MYDEBUG", "i=" + i + ", idx=" + idx + ", weight=" + temp + ", factor=" +
			// factor);

		}
		// Log.i("MYDEBUG", "-----");

		// average /= factor;
		average /= f.length;
		return average;
	}

	public float norm(float x, float y, float z)
	{
		return (float)Math.sqrt(x * x + y * y + z * z);
	}

	public float clamp(float v, float min, float max)
	{
		if (v > max)
			return max;
		else if (v < min)
			return min;
		else
			return v;
	}

	// =========================================
	// Point - simple class to hold an x-y point
	// =========================================

	private class Point
	{
		int x, y;

		Point(int xArg, int yArg)
		{
			x = xArg;
			y = yArg;
		}
	}

	// Low pass filter (smoothing algorithm) for sensor data.
	// See http://blog.thomnichols.org/2011/08/smoothing-sensor-data-with-a-low-pass-filter
	// More work is needed to decide on the best alpha. See comments above.
	protected float[] lowPass(float[] input, float[] output, float alpha)
	{
		for (int i = 0; i < input.length; i++)
			output[i] = output[i] + alpha * (input[i] - output[i]);
		return output;
	}

	// trim and round a float to the specified number of decimal places
	private float trim(float f, int decimalPlaces)
	{
		return ((int)(f * 10 * decimalPlaces + 0.5f)) / (float)(10 * decimalPlaces);
	}
	
	/*
	 * Screen updates are initiated in onFinish which executes every REFRESH_INTERVAL milliseconds
	 */
	public class ScreenRefreshTimer extends CountDownTimer
	{
		ScreenRefreshTimer(long millisInFuture, long countDownInterval)
		{
			super(millisInFuture, countDownInterval);
		}

		@Override
		public void onTick(long millisUntilFinished)
		{
		}

		@Override
		public void onFinish()
		{
			float tiltMagnitude = (float)Math.sqrt(pitch * pitch + roll * roll);
			float tiltAngle = tiltMagnitude == 0f ? 0f : (float)Math.asin(roll / tiltMagnitude) * RADIANS_TO_DEGREES;

			if (pitch > 0 && roll > 0)
				tiltAngle = 360f - tiltAngle;
			else if (pitch > 0 && roll < 0)
				tiltAngle = -tiltAngle;
			else if (pitch < 0 && roll > 0)
				tiltAngle = tiltAngle + 180f;
			else if (pitch < 0 && roll < 0)
				tiltAngle = tiltAngle + 180f;

			ep.updateTilt(tiltAngle, tiltMagnitude);

			// record the path of the ball (make sure the first target has been selected)
			if (targetsSelected > 0)
				actualPath.addElement(new Point((int)ep.xBallCenter, (int)ep.yBallCenter));

			// key selected stuff done here (perhaps move to separate method)
			if (ep.keySelected)
			{
				if (t1 == 0)
					t1 = System.currentTimeMillis();
				t2 = System.currentTimeMillis() - t1;

				Key k = ep.tentativeKey;
				++selectionCount;
				if (ep.targetSelected)
					++targetsSelected;

				sequenceEntered.append(k.label);
				sequenceEnteredField.setText(sequenceEntered);

				updateProgressText(targetsPerSequence, targetsSelected, selectionCount);

				samples.addElement(new Sample(t2, k.label)); // save time stamp and key label

				if (targetsSelected == targetsPerSequence)
					doEndOfSequence();

				ep.keySelected = false;
				if (ep.targetSelected)
				{
					ep.targetSelected = false;

					// ensure the target is not the same as the last target
					do
					{
						keyToSelect = r.nextInt(numberOfKeys);
					} while (keyToSelect == lastKeyToSelect);
					lastKeyToSelect = keyToSelect;

					ep.keyIndexToSelect = keyToSelect;
					desiredSequence.append(ep.KEY_LABELS.charAt(keyToSelect));
					minimumPath.addElement(new Point((int)(ep.key[keyToSelect].x), (int)(ep.key[keyToSelect].y)));
				}
			}
			ep.invalidate();
			this.start();
		}
	}
}