import york.*;

public class Tour
{
	private int N;
	private TSP tspInstance;

	// TSP solutions stored as permutation sequences
	private int[] best;
	private int[] current;
	private int[] temp;

	private int[] forwardEdge;
	private int[] backwardEdge;
	private int[] random;

	private int bestLength;
	private int currentLength;
	private int improvement;
	private int edge1, edge2;

	private MyStack stack;

	public Tour (TSP instance)
	{
		N = instance.getN();
		tspInstance = instance;
		best = new int[N];
		current = new int[N];
		temp = new int[N];
		random = new int[N];

		makeRandom();
		makeInsertion();

		stack = new MyStack(N);
		forwardEdge = new int[N];
		backwardEdge = new int[N];
	}

	private void makeRandom ()
	{
		for (int i = 0; i < N; i++)
			current[i] = i;

		for (int i = 0; i < N; i++)
		{
			int target = (int) (Math.random() * (N-i));
			swap(current, i, target);
		}
		currentLength = tspInstance.getLength(current);

		System.arraycopy(current, 0, random, 0, N);
		System.arraycopy(current, 0, best, 0, N);
		bestLength = currentLength;
	}

	private void makeInsertion ()
	{
		System.arraycopy(current, 0, temp, 0, N);
		current = new int[N];

		int i;
		for (i = 0; i < 3; i++)
			current[i] = temp[i];

		int nextInsert, minLocation;
		int minDistance, testDistance;

		for (i = 3; i < N; i++)
		{
			nextInsert = temp[i];
			minLocation = 0;
			minDistance = extraDistance(current[i-1], current[0], nextInsert);

			for (int j = 1; j < i; j++)
			{
				testDistance = extraDistance(current[j-1], current[j], nextInsert);
				if (testDistance < minDistance)
				{
					minDistance = testDistance;
					minLocation = j;
				}
			}

			for (int j = i; j > minLocation; j--)
			{
				current[j] = current[j-1];
			}
			current[minLocation] = nextInsert;
		}
		currentLength = tspInstance.getLength(current);

		System.arraycopy(current, 0, best, 0, N);
		bestLength = currentLength;
	}

	private int extraDistance (int first, int second, int insert)
	{
		int present = tspInstance.getDistance(first, second);
		int extra = tspInstance.getDistance(first, insert) +
			tspInstance.getDistance(insert, second);
		
		return extra - present;
	}

	public void twoOpt ()
	{
		twoOpt(null, 0.0);
	}

	public void twoOpt (Tour other, double saga)
	{
		int max, edges;

		double neighbourhood = Math.random();
		if (neighbourhood < 0.01)
			max = N/2;
		else if (neighbourhood < 0.03)
			max = N/4;
		else if (neighbourhood < 0.1)
			max = N/8;
		else if (neighbourhood < 0.25)
			max = N/16;
		else if (neighbourhood < 0.6)
			max = N/32;
		else
			max = N/64;

		if (other == null)
		{
			edge1 = (int) Math.floor(N*Math.random());
		}
		else
		{
			if (Math.random() < saga)
				edge1 = other.findDifferent(current);
			else
				edge1 = (int) Math.floor(N*Math.random());
		}
		edges = 2 + (int) Math.floor(max*Math.random());
		edge2 = edge1 + edges;

		if (edge2 >= N)
		{
			edge2 = edge1;
			edge1 = (edge1 + edges)%N;
		}
		calculateImprovement();
	}

	private void twoOpt (int[] array)
	{
		int index;
		for (index = edge1; index < edge2; index++)
			stack.push(array[index]);

		for (index = edge1; index < edge2; index++)
			array[index] = stack.pop();
	}

	public void makeEdges ()
	{
		for (int i = 1; i < N-1; i++)
		{
			forwardEdge[current[i]] = current[i+1];
			backwardEdge[current[i]] = current[i-1];
		}
		forwardEdge[current[0]] = current[1];
		forwardEdge[current[N-1]] = current[0];
		backwardEdge[current[0]] = current[N-1];
		backwardEdge[current[N-1]] = current[N-2];
	}

	public int findDifferent (int[] other)
	{
		int base = (int) Math.floor(N*Math.random());
		int index;
		int second, first;

		for (int i = 0; i < N; i++)
		{
			index = random[(base+i)%N];
			second = other[index];
			first = other[(index+(N-1))%N];
			if (forwardEdge[second] != first && backwardEdge[second] != first)
			{
				return index;
			}
		}
		return (int) Math.floor(N*Math.random());
	}

	public void update ()
	{
		twoOpt(current);
		currentLength -= improvement;

		if (currentLength <= bestLength)
		{
			bestLength = currentLength;
			System.arraycopy(current, 0, best, 0, N);
		}
	}

	public void backToBest ()
	{
		backToBest(false);
	}

	public void backToBest (boolean make)
	{
		System.arraycopy(best, 0, current, 0, N);
		currentLength = bestLength;
		if (make)
			makeEdges();
	}

	public void setBest ()
	{
		setBest(false);
	}

	public void setBest (boolean make)
	{
		System.arraycopy(current, 0, best, 0, N);
		bestLength = currentLength;
		if (make)
			makeEdges();
	}

	public int getCurrentLength ()
	{
		return currentLength;
	}

	public int getImprovement ()
	{
		return improvement;
	}

	private void calculateImprovement ()
	{
		int left1 = current[(edge1+(N-1))%N];
		int left2 = current[edge2-1];
		int right1 = current[edge1];
		int right2 = current[edge2];

		int currentDist = tspInstance.getDistance(left1, right1)
			+ tspInstance.getDistance(left2, right2);
		int newDist = tspInstance.getDistance(left1, left2)
			+ tspInstance.getDistance(right1, right2);

		improvement = currentDist - newDist;
	}

	private boolean validate ()
	{
		currentLength = tspInstance.getLength(current);

		boolean[] visited = new boolean[N];
		for (int i = 0; i < N; i++)
			visited[i] = false;

		for (int i = 0; i < N; i++)
			visited[current[i]] = true;

		for (int i = 0; i < N; i++)
			if (!visited[i])
				return false;
		return true;
	}

	private void swap (int[] array, int first, int second)
	{
		int temp = array[first];
		array[first] = array[second];
		array[second] = temp;
	}

	public void loadTour (String file)
	{
		YorkReader reader = new YorkReader(file);
		reader.readInt();

		for (int i = 0; i < N; i++)
			current[i] = reader.readInt()-1;

		reader.close();

		currentLength = tspInstance.getLength(current);
	}

	public void printTour ()
	{
		System.out.println("Valid tour = " + validate());
		System.out.println("Length = " + currentLength);
/*
		System.out.println("Cities");
		for (int i = 0; i < N; i++)
		{
			System.out.println(current[i]);
		}
*/
	}

	public void printTour (String file)
	{
		YorkWriter writer = new YorkWriter(file);

		writer.println("Valid tour = " + validate());
		writer.println("Length = " + currentLength);
		writer.println("Cities");
		for (int i = 0; i < N; i++)
		{
			writer.println(current[i]);
		}
		writer.close();
	}
}


class MyStack
{
	private int curr;
	private int[] stack;

	public MyStack (int N)
	{
		stack = new int[N];
	}

	public void push (int value)
	{
		stack[curr] = value;
		curr++;
	}

	public int pop ()
	{
		curr--;
		return stack[curr];
	}
}