      SUBROUTINE HARDCOPY_RANGE(XMINH,XMAXH,YMINH,YMAXH,
     *                          XMINHP,XMAXHP,YMINHP,YMAXHP,IORIENTH,*)
C======================================================================C
C                                                                      C
C  HARDCOPY_RANGE performs a windowing transformation of the world     C
C  coordinate system (user units) to a viewport on the internal        C
C  (hardcopy, printronix) bit map.  It also updates the                C
C  PLOT-to-MONITOR and MONITOR-to-PLOT transformations, but NOT the    C
C  PLOT-to-MONITOR2 and MONITOR2-to-PLOT transformations.  Thus, if    C
C  a second monitor (MONITOR2) is being used for graphics display, be  C
C  sure to call MONITOR2_RANGE after calling HARDCOPY_RANGE !          C
C                                                                      C
C  More specifically, HARDCOPY_RANGE defines the range of the hardcopy C
C  viewport:                                                           C
C      XMINH <= XH <= XMAXH; YMINH <= YH <= YMAXH (dots)               C
C  within the printronix page in dots:                                 C
C      0 <= XH <= 2047 (vertical); 0 <= YH <= 785 (horizontal)         C
C  (The origin of the printronix page (XH,YH) = (0,0) is in the upper  C
C  left hand corner of the page).  It also defines the plot coordinate C
C  range (the window) that is mapped into the hardcopy viewport:       C
C       XMINHP <= X <= XMAXHP; YMINHP <= Y <= YMAXHP,                  C
C  as well as the orientation of the plot on the hardcopy page given   C
C  by IORIENTH:                                                        C
C            IORIENTH =  1: X is vertical; Y is horizontal;            C
C                           (XMINHP,YMINHP) appears in the             C
C                           upper left hand corner of the              C
C                           hardcopy viewport.                         C
C                     =  0: no hardcopy will be plotted in the         C
C                           hardcopy bitmap, although the above        C
C                           ranges will still be set and used          C
C                           for plotting on the monitor.               C
C                     = -1: X is horizontal; Y is vertical;            C
C                           (XMINHP,YMINHP) appears in the             C
C                           lower left hand corner of the              C
C                           hardcopy viewport.                         C
C                                                                      C
C  The values of XMINH,XMAXH,YMINH,YMAXH are restricted to the range   C
C  of the printronix page in dots:                                     C
C       0 <= XH <= 2047; 0 <= YH <= 785.                               C
C  If they fall outside this range they will be set to the nearest     C
C  limit.  The values of XMINHP,XMAXHP,YMINHP,YMAXHP are not           C
C  restricted.  If any of the sides of the "HARDCOPY" viewport or "PLOT"
C  window are equal to 0 (eg. XMINH=XMAXH), then an error message is   C
C  printed on unit 6 and a RETURN1 is performed.                       C
C                                                                      C
C       Note: Within the plot package there are 4 distinct             C
C             coordinate systems used:                                 C
C                                                                      C
C  First, there is the "HARDCOPY" coordinate system "(XH,YH)" with:    C
C             0 <= XH <= 2047 (vertical on the printronix)             C
C             0 <= YH <= 785  (horizontal on the printronix).          C
C  This coordinate system never changes and its units are printronix   C
C  dots.                                                               C
C                                                                      C
C  Second, there is the user defined "PLOT" coordinate system "(X,Y)"  C
C  with                                                                C
C             XMINHP <= X <= XMAXHP; YMINHP <= Y <= YMAXHP             C
C                                                                      C
C  Third, there is the "MONITOR" coordinate system "(XM,YM)" with      C
C             0 <= XM <= XMAXMSIZE (horizontal on the screen)          C
C             0 <= YM <= YMAXMSIZE (vertical on the screen).           C
C  This coordinate system is fixed for each type of monitor and its    C
C  units are monitor dots.                                             C
C                                                                      C
C  Fourth, there is the "MONITOR2" coordinate system defined           C
C  analogously to the MONITOR coordinates.                             C
C                                                                      C
C  The following coordinate transformations tie the first three        C
C  coordinate systems together:                                        C
C                                                                      C
C             /PLOT_TO_HARDCOPY/ RMATRIXHP(2,2),SHIFTHP(2)             C
C             |XH|               |X|                                   C
C             |  | = RMATRIXHP * | | + SHIFTHP                         C
C             |YH|               |Y|                                   C
C                                                                      C
C             /PLOT_TO_MONITOR/  RMATRIXMP(2,2),SHIFTMP(2)             C
C             |XM|               |X|                                   C
C             |  | = RMATRIXMP * | | + SHIFTMP                         C
C             |YM|               |Y|                                   C
C                                                                      C
C             /MONITOR_TO_PLOT/  RMATRIXPM(2,2),SHIFTPM(2)             C
C             |X|               |XM|                                   C
C             | | = RMATRIXPM * |  | + SHIFTPM                         C
C             |Y|               |YM|                                   C
C                                                                      C
C             /HARDCOPY_TO_MONITOR/  RMATRIXMH(2,2),SHIFTMH(2)         C
C             |XM|               |XH|                                  C
C             |  | = RMATRIXMH * |  | + SHIFTMH                        C
C             |YM|               |YH|                                  C
C                                                                      C
C   Written by Arthur Haynes, TRIUMF U.B.C., July 14, 1982.            C
C                                                                      C
C   Input  Parameters: XMINH,XMAXH,YMINH,YMAXH,XMINHP,XMAXHP,          C
C                      YMINHP,YMAXHP (R*4); IORIENTH (I*4).            C
C                                                                      C
C   If HARDCOPY_RANGE is never called these parameters default         C
C   to the following values:                                           C
C   0=XMINH <= XH <= XMAXH=479; 0=YMINH <= YH <= YMAXH=639;            C
C   0=XMINHP <= X <= XMAXHP=639; 0=YMINHP <= Y <= YMAXHP=479;          C
C   IORIENTH = -1.                                                     C
C                                                                      C
C   Modified Jan 19/88 by F.W. Jones:  The internal bitmap is now      C
C   stored with eight significant bits per byte instead of six, so     C
C   the maximum dimensions of the bitmap have been increased to        C
C   1504 x 2047.                                                       C
C                                                                      C
C   Modified Dec 05/89 by F.W. Jones:  support added for HP Laserjet   C
C   300 dpi hardcopies using the dynamic bitmap.  Also fixed a bug:    C
C   MAXLINES should not be set when the dynamic bitmap is being used.  C
C                                                                      C
C   Modified Jan 04/90 by J.Chuma:  support added for HP Laserjet      C
C   compression.                                                       C
C======================================================================C
      COMMON /HARDCOPYRANGE/ XMINHP2,XMAXHP2,YMINHP2,YMAXHP2,IORIENTH2
      COMMON /HARDCOPYRANGE2/ XMINH2,XMAXH2,YMINH2,YMAXH2
      COMMON /MONITORRANGE/ XMINMP,XMAXMP,YMINMP,YMAXMP,XMINM,XMAXM,
     *                       YMINM,YMAXM
      COMMON /MONITOR_SIZE/ XMINMSIZE,XMAXMSIZE,YMINMSIZE,YMAXMSIZE
      COMMON /PLOT_TO_MONITOR/ RMATRIXMP(2,2),SHIFTMP(2)
      COMMON /MONITOR_TO_PLOT/ RMATRIXPM(2,2),SHIFTPM(2)
      COMMON /HARDCOPY_TO_MONITOR/ RMATRIXMH(2,2),SHIFTMH(2)
      COMMON /PLOT_TO_HARDCOPY/ RMATRIXHP(2,2),SHIFTHP(2)
      COMMON /CALL_MONITOR_RANGE/ MONITOR_RANGE_CALLED
      LOGICAL MONITOR_RANGE_CALLED
C
      COMMON/BITMAP_STATUS/BITMAP_EMPTY,MAXLINES
      LOGICAL BITMAP_EMPTY
      INTEGER MAXLINES
C
      COMMON/GRAPHICS_BITMAP/HPAINT,NBYTES,IOFFSET,NXDIM,NYDIM,MAXY
      LOGICAL*1 HPAINT(4)
      COMMON /BITMAP_DEVICE/ IBIT

      DATA XMINHSIZE,XMAXHSIZE,YMINHSIZE,YMAXHSIZE/0.,2047.,0.,1504./

C==================================================================
C==   Make sure that the hardcopy viewport is within the bitmap  ==
C==   dimensions.                                                ==
C==================================================================
      IF(IBIT.NE.5 .AND. IBIT.NE.32 .AND. IBIT.NE.132)THEN
        XMINH2=MAX(MIN(XMINH,XMAXHSIZE),XMINHSIZE)
        XMAXH2=MAX(MIN(XMAXH,XMAXHSIZE),XMINHSIZE)
        YMINH2=MAX(MIN(YMINH,YMAXHSIZE),YMINHSIZE)
        YMAXH2=MAX(MIN(YMAXH,YMAXHSIZE),YMINHSIZE)
      ELSE IF( IBIT .EQ. 5 )THEN             ! HP PaintJet
        XMINH2=MAX(MIN(XMINH,9899.),0.)   ! 9900=180dpi*11in*5pgs
        XMAXH2=MAX(MIN(XMAXH,9899.),0.)
        YMINH2=MAX(MIN(YMINH,1439.),0.)   ! 1440=180dpi*8in*1pg
        YMAXH2=MAX(MIN(YMAXH,1439.),0.)
C Find array dimensions NXDIM and NYDIM:
        NXMAX=NINT(MAX(XMINH2,XMAXH2))+1     !#dots down page
        NYMAX=NINT(MAX(YMINH2,YMAXH2))+1     !#dots across page
        NYDIM=NXMAX*3                        !usually 5400
        NXDIM=NYMAX/8                        !usually 180
        IF(MOD(NYMAX,8).NE.0)NXDIM=NXDIM+1
      ELSE IF(IBIT.EQ.32 .OR. IBIT.EQ.132)THEN    !HP Laserjet 300dpi
        XMINH2=MAX(MIN(XMINH,2999.),0.)   ! 3000=300dpi*10in
        XMAXH2=MAX(MIN(XMAXH,2999.),0.)
        YMINH2=MAX(MIN(YMINH,2249.),0.)   ! 2250=300dpi*7.5in
        YMAXH2=MAX(MIN(YMAXH,2249.),0.)
C Find array dimensions NXDIM and NYDIM:
        NXMAX=NINT(MAX(XMINH2,XMAXH2))+1     !#dots down page
        NYMAX=NINT(MAX(YMINH2,YMAXH2))+1     !#dots across page
        NYDIM=NXMAX                          !usually 3000
        NXDIM=NYMAX/8                        !usually 282
        IF(MOD(NYMAX,8).NE.0)NXDIM=NXDIM+1
      ENDIF

C==================================================================
C==   Make sure that none of the sides of the viewport and       ==
C==   window are 0.                                              ==
C==================================================================
      IF(XMINH2.EQ.XMAXH2)THEN
         WRITE(6,1)
1        FORMAT(' ***Error*** in HARDCOPY_RANGE: XMINH = XMAXH.')
         RETURN1
      END IF
      IF(YMINH2.EQ.YMAXH2)THEN
         WRITE(6,2)
2        FORMAT(' ***Error*** in HARDCOPY_RANGE: YMINH = YMAXH.')
         RETURN1
      END IF
      IF(XMINHP.EQ.XMAXHP)THEN
         WRITE(6,3)
3        FORMAT(' ***Error*** in HARDCOPY_RANGE: XMINHP = XMAXHP.')
         RETURN1
      END IF
      IF(YMINHP.EQ.YMAXHP)THEN
         WRITE(6,4)
4        FORMAT(' ***Error*** in HARDCOPY_RANGE: YMINHP = YMAXHP.')
         RETURN1
      END IF
      XMINHP2=XMINHP
      XMAXHP2=XMAXHP
      YMINHP2=YMINHP
      YMAXHP2=YMAXHP
      IORIENTH2=IORIENTH
      IF(IORIENTH2.NE.0.AND.IORIENTH2.NE.-1)IORIENTH2=1
C Record maximum number of lines in bitmap since last clear
C (not for dynamic bitmap)
      IF(IBIT.NE.5 .AND. IBIT.NE.32 .AND. IBIT.NE.132)
     & MAXLINES=MAX(MAXLINES,INT(XMAXH2)+1)

C=================================================================
C==   Calculate the "PLOT" to "HARDCOPY" transformation.        ==
C==                                                             ==
C==   FOR IORIENTH = 1:                                         ==
C==   XH = (X-XMINHP)/(XMAXHP-XMINHP)*(XMAXH-XMINH)+XMINH       ==
C==      = RMATRIXHP(1,1)*X+RMATRIXHP(1,2)*Y+SHIFTHP(1).        ==
C==   YH = (Y-YMINHP)/(YMAXHP-YMINHP)*(YMAXH-YMINH)+YMINH       ==
C==      = RMATRIXHP(2,1)*X+RMATRIXHP(2,2)*Y+SHIFTHP(2).        ==
C==                                                             ==
C==   FOR IORIENTH = -1:                                        ==
C==   XH = XMAXH-(Y-YMINHP)/(YMAXHP-YMINHP)*(XMAXH-XMINH)       ==
C==      = RMATRIXHP(1,1)*X+RMATRIXHP(1,2)*Y+SHIFTHP(1).        ==
C==   YH = (X-XMINHP)/(XMAXHP-XMINHP)*(YMAXH-YMINH)+YMINH       ==
C==      = RMATRIXHP(2,1)*X+RMATRIXHP(2,2)*Y+SHIFTHP(2).        ==
C=================================================================
      IF(IORIENTH2.EQ.-1)GO TO 10
      RMATRIXHP(1,1)=(XMAXH2-XMINH2)/(XMAXHP2-XMINHP2)
      RMATRIXHP(1,2)=0.
      RMATRIXHP(2,1)=0.
      RMATRIXHP(2,2)=(YMAXH2-YMINH2)/(YMAXHP2-YMINHP2)
      SHIFTHP(1)=XMINH2-XMINHP2*RMATRIXHP(1,1)
      SHIFTHP(2)=YMINH2-YMINHP2*RMATRIXHP(2,2)
      GO TO 20
10    RMATRIXHP(1,1)=0.
      RMATRIXHP(1,2)=-(XMAXH2-XMINH2)/(YMAXHP2-YMINHP2)
      RMATRIXHP(2,1)=(YMAXH2-YMINH2)/(XMAXHP2-XMINHP2)
      RMATRIXHP(2,2)=0.
      SHIFTHP(1)=XMAXH2-YMINHP2*RMATRIXHP(1,2)
      SHIFTHP(2)=YMINH2-XMINHP2*RMATRIXHP(2,1)
20    IF(MONITOR_RANGE_CALLED)GO TO 40
C=================================================================
C==   The MONITOR_RANGE subroutine has never been called.       ==
C==   Hence the monitor should display the whole range of the   ==
C==   hardcopy viewport specified by HARDCOPY_RANGE.            ==
C==   Therefore the default values of the transformation matrix ==
C==   RMATRIXMH and the translation vector SHIFTMH must be      ==
C==   changed in order to accomplish this.                      ==
C==                                                             ==
C==   Calculate the "HARDCOPY" to "MONITOR" transformation.     ==
C==                                                             ==
C==   FOR IORIENTH = IORIENTM = 1:                              ==
C==   XM = (XH-XMINH)/(XMAXH-XMINH)*(XMAXM-XMINM)+XMINM         ==
C==      = RMATRIXMH(1,1)*XH+RMATRIXMH(1,2)*YH+SHIFTMH(1).      ==
C==   YM = (YH-YMINH)/(YMAXH-YMINH)*(YMAXM-YMINM)+YMINM         ==
C==      = RMATRIXMH(2,1)*XH+RMATRIXMH(2,2)*YH+SHIFTMH(2).      ==
C==                                                             ==
C==   FOR IORIENTH = IORIENTM = -1:                             ==
C==   XM = (YH-YMINH)/(YMAXH-YMINH)*(XMAXM-XMINM)+XMINM         ==
C==      = RMATRIXMH(1,1)*XH+RMATRIXMH(1,2)*YH+SHIFTMH(1).      ==
C==   YM = YMAXM-(XH-XMINH)/(XMAXH-XMINH)*(YMAXM-YMINM)         ==
C==        RMATRIXMH(2,1)*XH+RMATRIXMH(2,2)*YH+SHIFTMH(2).      ==
C=================================================================
      IF(IORIENTH2.EQ.-1)GO TO 30
      RMATRIXMH(1,1)=(XMAXM-XMINM)/(XMAXH2-XMINH2)
      RMATRIXMH(1,2)=0.
      RMATRIXMH(2,1)=0.
      RMATRIXMH(2,2)=(YMAXM-YMINM)/(YMAXH2-YMINH2)
      SHIFTMH(1)=XMINM-RMATRIXMH(1,1)*XMINH2
      SHIFTMH(2)=YMINM-RMATRIXMH(2,2)*YMINH2
      GO TO 40
30    RMATRIXMH(1,1)=0.
      RMATRIXMH(1,2)= (XMAXM-XMINM)/(YMAXH2-YMINH2)
      RMATRIXMH(2,1)=-(YMAXM-YMINM)/(XMAXH2-XMINH2)
      RMATRIXMH(2,2)=0.
      SHIFTMH(1)=XMINM-RMATRIXMH(1,2)*YMINH2
      SHIFTMH(2)=YMAXM-RMATRIXMH(2,1)*XMINH2
C=================================================================
C==   Calculate the transformation matrix RMATRIXMP from the    ==
C==   "PLOT" coordinate system to the "MONITOR" coordinate      ==
C==   system:                                                   ==
C==                                                             ==
C==   RMATRIXMP      = RMATRIXMH          * RMATRIXHP           ==
C==                                                             ==
C==   MONITOR<--PLOT = MONITOR<--HARDCOPY * HARDCOPY<--PLOT     ==
C=================================================================
40    RMATRIXMP(1,1)=RMATRIXMH(1,1)*RMATRIXHP(1,1)+
     *               RMATRIXMH(1,2)*RMATRIXHP(2,1)
      RMATRIXMP(1,2)=RMATRIXMH(1,1)*RMATRIXHP(1,2)+
     *               RMATRIXMH(1,2)*RMATRIXHP(2,2)
      RMATRIXMP(2,1)=RMATRIXMH(2,1)*RMATRIXHP(1,1)+
     *               RMATRIXMH(2,2)*RMATRIXHP(2,1)
      RMATRIXMP(2,2)=RMATRIXMH(2,1)*RMATRIXHP(1,2)+
     *               RMATRIXMH(2,2)*RMATRIXHP(2,2)
C=================================================================
C==   Calculate the translation vector SHIFTMP from the "PLOT"  ==
C==   coordinate system to the "MONITOR" coordinate system.     ==
C==                                                             ==
C==   SHIFTMP = RMATRIXMH * SHIFTHP + SHIFTMH                   ==
C=================================================================
      SHIFTMP(1)=RMATRIXMH(1,1)*SHIFTHP(1)+RMATRIXMH(1,2)*SHIFTHP(2)+
     *           SHIFTMH(1)
      SHIFTMP(2)=RMATRIXMH(2,1)*SHIFTHP(1)+RMATRIXMH(2,2)*SHIFTHP(2)+
     *           SHIFTMH(2)
C=================================================================
C==   Calculate the inverse transformation matrix RMATRIXPM from==
C==   the "MONITOR" coordinate system to the "PLOT" coordinate  ==
C==   system:                                                   ==
C==                                                             ==
C==   RMATRIXPM      = RMATRIXMP**-1                            ==
C==                                                             ==
C==   PLOT<--MONITOR = (MONITOR<--PLOT)**-1                     ==
C=================================================================
      DETMP=RMATRIXMP(1,1)*RMATRIXMP(2,2)-RMATRIXMP(2,1)*RMATRIXMP(1,2)
      RMATRIXPM(1,1)=RMATRIXMP(2,2)/DETMP
      RMATRIXPM(1,2)=-RMATRIXMP(1,2)/DETMP
      RMATRIXPM(2,1)=-RMATRIXMP(2,1)/DETMP
      RMATRIXPM(2,2)=RMATRIXMP(1,1)/DETMP
C=================================================================
C==   Calculate the inverse translation vector SHIFTPM from the ==
C==   "MONITOR" coordinate system to the "PLOT" coordinate      ==
C==   system.                                                   ==
C==                                                             ==
C==   SHIFTPM = -RMATRIXPM * SHIFTMP                            ==
C=================================================================
      SHIFTPM(1)=-RMATRIXPM(1,1)*SHIFTMP(1)-RMATRIXPM(1,2)*SHIFTMP(2)
      SHIFTPM(2)=-RMATRIXPM(2,1)*SHIFTMP(1)-RMATRIXPM(2,2)*SHIFTMP(2)
C=================================================================
C==   Calculate the boundaries in the "PLOT" coordinate system  ==
C==   corresponding to the monitor viewport.                    ==
C==   Expand these boundaries by 1/1000000 of their original    ==
C==   dimensions in order to avoid round off problems when one  ==
C==   plots right on the boundaries.                            ==
C=================================================================
      XMINMP2=RMATRIXPM(1,1)*XMINM+RMATRIXPM(1,2)*YMINM+SHIFTPM(1)
      YMINMP2=RMATRIXPM(2,1)*XMINM+RMATRIXPM(2,2)*YMINM+SHIFTPM(2)
      XMAXMP2=RMATRIXPM(1,1)*XMAXM+RMATRIXPM(1,2)*YMAXM+SHIFTPM(1)
      YMAXMP2=RMATRIXPM(2,1)*XMAXM+RMATRIXPM(2,2)*YMAXM+SHIFTPM(2)
      XMINMP=XMINMP2-.000001*(XMAXMP2-XMINMP2)
      XMAXMP=XMAXMP2+.000001*(XMAXMP2-XMINMP2)
      YMINMP=YMINMP2-.000001*(YMAXMP2-YMINMP2)
      YMAXMP=YMAXMP2+.000001*(YMAXMP2-YMINMP2)
      RETURN
      END
