      SUBROUTINE DLINE(X,Y,LTYPE,IEND)
C======================================================================C
C                                                                      C
C  DLINE                                       F.W. Jones, TRIUMF      C
C                                                                      C
C  Draws a line of specified line type from the current location       C
C  to location (X,Y).                                                  C
C                                                                      C
C  The line types are in four different styles:                        C
C                                                                      C
C  1) Ordinary solid line                                              C
C  2) Double line of specified width                                   C
C  3) Dashed line with specified dash and space lengths                C
C  4) Dashed line with two different dash lengths                      C
C                                                                      C
C  Input:                                                              C
C                                                                      C
C    X,Y     Coordinates in user units.                                C
C                                                                      C
C    LTYPE   Number of the line type to be used (1-10).                C
C                                                                      C
C    IEND    If IEND=1, the dashing sequence is ended at (X,Y)         C
C            and a new sequence will begin on the next call.           C
C                                                                      C
C            If IEND=2, the dashing sequence is ended at (X,Y)         C
C            and if the line does not end in a dash, the last dash     C
C            will be extended to (X,Y).                                C
C                                                                      C
C            Other values of IEND are ignored.                         C
C                                                                      C
C                                                                      C
C  For each line type LTYPE, the appearance of the line is determined  C
C  by three parameters P1,P2,P3 that are stored in an internal table.  C
C                                                                      C
C  If P1 = 0, the line is an ordinary solid line.                      C
C                                                                      C
C  If P1 > 0 and P2 = 0, the line is a double line of width P1         C
C                                                                      C
C  If P1 > 0 and P2 > 0, the line is a dashed line, with dash          C
C  length P1 and space length P2.                                      C
C                                                                      C
C  If P1 > 0 and P2 > 0 and P3 > 0, the line is a dashed line,         C
C  with first dash length P1, space length P2, and second dash         C
C  length P3.                                                          C
C                                                                      C
C  NOTE:  The above quantities refer to the user coordinate system     C
C         established by HARDCOPY_RANGE.                               C
C                                                                      C
C  The user may define his own line types, or he may use the default   C
C  types established in this routine, which are suitable for the d     C
C  default 640 x 480 coordinate system.  Line types are user-defined   C
C  by calling entry point DLINESET:                                    C
C                                                                      C
C      CALL DLINESET(LTYPE,P1,P2,P3)                                   C
C                                                                      C
C  where LTYPE is a line type number between 1 and 10 and P1-P3        C
C  are the desired parameters.  Alternatively, the default line types  C
C  can be scaled to fit a user coordinate system by calling:           C
C                                                                      C
C      CALL DLINESCALE(LTYPE,SF)                                       C
C                                                                      C
C  where SF is the desired scale factor.  For example, if the user     C
C  coordinate system is 10 units in X by 7.5 units in Y, the scale     C
C  factor SF=10/640 would be appropriate.                              C
C                                                                      C
C  This routine always plots from the current location, which is       C
C  generally determined by the last call to PLOT_R.  Thus a call       C
C  to PLOT_R with pen up would usually precede a series of calls       C
C  to DLINE.  In order to plot a continuous dashed line through        C
C  a series of points, DLINE "remembers" where it's at between         C
C  calls.  As long as the current location is the same as that         C
C  where DLINE left off on the previous call, and the line type        C
C  LTYPE has not changed, DLINE will continue dashing from where it    C
C  left off.  Thus the dashing may be interrupted, a symbol or other   C
C  items plotted, and the dashing can be continued by calling PLOT_R   C
C  with pen up to the old location before calling DLINE.               C
C                                                                      C
C  Modified Sept 27/84 for output to EDGR drawing files.               C
C                                                                      C
C  Modified May 13/86: DLINESET and DLINESCALE have been incorporated  C
C  as entry points so that the line table is accessed directly rather  C
C  than through COMMON.                                                C
C                                                                      C
C  Modified Sept 21/87: the algorithm has be re-written in terms of    C
C  a distance parameter which improves the drawing accuracy and        C
C  simplifies the code.                                                C
C                                                                      C
C  Modified Jan 23/90: if the STROKE flag is set, output to the DWG    C
C  file is done through PLOT_R rather than DLINE.                      C
C                                                                      C
C======================================================================C

C Line table:
      COMMON/LINE_TABLE/DLINTAB(10,3)
C Line type        1  2   3   4   5   6   7   8   9  10
      DATA DLINTAB/0.,2.,30.,30.,20.,20.,10.,10., 3., 5.,   !P1
     &             0.,0.,10.,15.,10., 8., 6., 6.,10.,15.,   !P2
     &             0.,0., 0.,10., 0., 6., 0., 4., 0., 0./   !P3

C Default values for line table parameters:
      REAL DEFAULT(10,3)
C Line type        1  2   3   4   5   6   7   8   9  10
      DATA DEFAULT/0.,2.,30.,30.,20.,20.,10.,10., 3., 5.,   !P1
     &             0.,0.,10.,15.,10., 8., 6., 6.,10.,15.,   !P2
     &             0.,0., 0.,10., 0., 6., 0., 4., 0., 0./   !P3

C Common with PLOT_R:
      COMMON/PLOT_PREVIOUS/XP,YP,IPENP

C Graphics Editor COMMON:
      COMMON/CDWG/DWGON,DWGTXT,LDWG,LDWT,IRECG,IRECT,STROKE
      LOGICAL DWGON,DWGTXT,STROKE

C Force a "begin dashing" condition on first call:
      INTEGER LTYPOLD/0/

      REAL DD(4)

C======================================================================C
C  Procedure begins.
C======================================================================C

C Draw nothing if we are going nowhere:
      IF(X.EQ.XP.AND.Y.EQ.YP)RETURN

      LTYP=LTYPE
C Illegal LTYP: default to solid line:
      IF(LTYP.LT.1.OR.LTYP.GT.10)LTYP=1

C Encode operation to drawing file if open:
      IF(DWGON.AND..NOT.STROKE)THEN
        CALL PUT_DWG(X,Y,2,LTYP)
        DWGTXT=.TRUE.      !inhibit encoding from PLOT_R
      ENDIF

C Get parameters:
      P1=DLINTAB(LTYP,1)
      P2=DLINTAB(LTYP,2)
      P3=DLINTAB(LTYP,3)

C Draw an ordinary vector:
      IF(P1.EQ.0.)THEN
        CALL PLOT_R(X,Y,2)
        GO TO 999
      ENDIF

C Useful stuff:
      DELTX=X-XP
      DELTY=Y-YP
      DIST=SQRT(DELTX*DELTX + DELTY*DELTY)
C Sine and cosine of orientation angle:
      SINTH=DELTY/DIST
      COSTH=DELTX/DIST

C======================================================================C
C  Double line
C======================================================================C

      IF(P2.EQ.0.)THEN
        HWID=P1/2.
        DX=HWID*SINTH
        DY=HWID*COSTH
        XLOC=XP
        YLOC=YP
        CALL PLOT_R(XLOC-DX,YLOC+DY,3)
        CALL PLOT_R(X-DX,Y+DY,2)
        CALL PLOT_R(XLOC+DX,YLOC-DY,3)
        CALL PLOT_R(X+DX,Y-DY,2)
C Do penup to proper location for continuation:
        CALL PLOT_R(X,Y,3)
        GO TO 999
      ENDIF

C======================================================================C
C  Dashed line
C======================================================================C

C Distance increments for operation loop:
      IF(P3.EQ.0.)P3=P1
      DD(1)=P1
      DD(2)=P2
      DD(3)=P3
      DD(4)=P2
C Start point:
      X0=XP
      Y0=YP

C Continuing previous line?
      IF(LTYP.EQ.LTYPOLD.AND.X0.EQ.XOLD.AND.Y0.EQ.YOLD)THEN
        IF(D.GT.0.)GO TO 210      !continuing dash or space
        GO TO 200
      ENDIF

C Begin dashing sequence:
      IOP=0        !operation number
      D=0.         !distance travelled

C======================================================================C
C  Operation loop begins
C======================================================================C

200   IOP=IOP+1
      IF(IOP.EQ.5)IOP=1
      D=D+DD(IOP)

210   XLOC=X0+D*COSTH
      YLOC=Y0+D*SINTH
      IF(IOP.EQ.1.OR.IOP.EQ.3)THEN
        IPEN=2
      ELSE
        IPEN=3
      ENDIF

C Last operation:
      IF(D.GE.DIST)THEN
        XLOC=X
        YLOC=Y
C Extend last dash?
        IF(IEND.EQ.2.AND.(IOP.EQ.2.OR.IOP.EQ.4))IPEN=2
      ENDIF

      CALL PLOT_R(XLOC,YLOC,IPEN)

      IF(D.LT.DIST)GO TO 200      !do next operation

999   LTYPOLD=LTYP      !save line type
      IF(IEND.EQ.1.OR.IEND.EQ.2)LTYPOLD=0      !no continuation
      XOLD=X      !save end point
      YOLD=Y
      D=D-DIST      !portion of dash or space to be continued
      DWGTXT=.FALSE.
      RETURN      !end of DLINE

      ENTRY DLINESET(LTYP_SET,P1_SET,P2_SET,P3_SET)
C======================================================================C
C
C  DLINESET                                      F.W. Jones, TRIUMF
C
C  User interface to set up line type characteristics for DLINE.
C
C  If LTYP>0, parameters P1,P2,P3 are set for line type LTYP.
C  If LTYP<0, parameters P1,P2,P3 are returned for line type -LTYP.
C  If LTYP=0, parameters for all line types are reset to defaults.
C
C======================================================================C

      IF(ABS(LTYP_SET).GT.10)RETURN
      IF(LTYP_SET.GT.0)THEN
        DLINTAB(LTYP_SET,1)=P1_SET
        DLINTAB(LTYP_SET,2)=P2_SET
        DLINTAB(LTYP_SET,3)=P3_SET
      ELSE IF(LTYP_SET.LT.0)THEN
        P1_SET=DLINTAB(-LTYP_SET,1)
        P2_SET=DLINTAB(-LTYP_SET,2)
        P3_SET=DLINTAB(-LTYP_SET,3)
      ELSE
        DO I=1,10
          DO J=1,3
            DLINTAB(I,J)=DEFAULT(I,J)
          ENDDO
        ENDDO
      ENDIF
      RETURN

      ENTRY DLINESCALE(LTYP_SCALE,SF)
C======================================================================C
C
C  DLINESCALE                                      F.W. Jones, TRIUMF
C
C  User interface to scale the line table parameters by factor SF.
C
C  If LTYP>0, parameters P1,P2,P3 are scaled for line type LTYP.
C  If LTYP=0, parameters for all line types are scaled.
C
C======================================================================C

      IF(LTYP_SCALE.GT.10.OR.LTYP_SCALE.LT.0)RETURN

C Range of line types to be scaled:
      L1=LTYP_SCALE
      L2=LTYP_SCALE
      IF(LTYP_SCALE.EQ.0)THEN
        L1=1
        L2=10
      ENDIF

      DO I=L1,L2
        DO J=1,3
          DLINTAB(I,J)=DLINTAB(I,J)*SF
        ENDDO
      ENDDO
      RETURN
      END
